<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class IndexRxLenses extends Command
{
    protected $signature = 'rx:index-lenses';
    protected $description = 'Index Rx Validations into Redis with inverted indexes';

    public function handle()
    {
        $this->info("🔄 Starting Rx Lenses indexing…");

        $redis = Cache::getRedis();
        $cacheTag = "rx_lenses";

        // Clear all old caches
        Cache::tags([$cacheTag])->flush();
        clearCustomCache('rx');
        $this->info("🧹 Cleared old cache…");

        // Fields we want to index
        $indexFields = [
            'lens_type',
            'lens_name',
            'lens_material',
            'lens_colour',
            'lens_filter',
            'coating_name',
            'coating_made',
            'coating_colour',
            'lens_manufacturer',
            'sub_type',
            'base_curve',
        ];

        $chunkSize = 1000;
        $count = 0;

        DB::table('rx_validations')->orderBy('id')->chunk($chunkSize, function ($rows) use ($redis, $indexFields, &$count) {
            foreach ($rows as $row) {
                $rowId = $row->id;
                $rowArr = (array) $row;

                // Save full row JSON
                $redis->set("rx:rows:{$rowId}", json_encode($rowArr));

                // Build inverted indexes
                foreach ($indexFields as $field) {
                    if (!empty($rowArr[$field])) {
                        $val = trim($rowArr[$field]);

                        if ($val !== '') {
                            $key = "rx_index:{$field}:{$val}";
                            $redis->sadd($key, $rowId);
                        }
                    }
                }

                $count++;
                if ($count % 5000 === 0) {
                    $this->info("✅ Indexed {$count} rows…");
                }
            }
        });

        $this->info("🎉 Finished indexing {$count} rows.");
    }
}