<?php

namespace App\Console\Commands;

use App\Custom\DropboxComponent;
use App\Models\Backend\Order;
use App\Models\Backend\RxOrders;
use App\Models\Backend\StockOrder;
use Illuminate\Console\Command;
use PDF;

class uploadDropboxFiles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'upload:dropboxfiles {id : The order ID}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Upload order PDF to Dropbox';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $id = $this->argument('id');
        $order = Order::find($id);

        if (!$order) {
            $this->error('Order not found');
            return;
        }

        // Generate PDF similar to printPdf method
        $html = '';
        switch ($order->type) {
            case Order::ORDER_RX:
                $user = $order->user;
                $profile = $order->user->profile;
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                $rxOrderArray = $rxOrder->getShowData();
                $rawData = json_decode($rxOrder->data, true);
                $shouldWait = $this->waitOrder($rawData);
                $path = 'backend.auth.orders.includes.print-pdf.rx';
                $data = [
                    'order' => $order,
                    'rxOrder' => $rxOrder,
                    'user' => $user,
                    'profile' => $profile,
                    'rxOrderArray' => $rxOrderArray,
                    'shouldWait' => $shouldWait,
                ];
                break;
            case Order::ORDER_BARCODE:
            case Order::ORDER_STOCK:
                $stockOrder = StockOrder::where('order_id', $order->id)->first();
                $user = $order->user;
                $profile = $order->user->profile;
                $path = 'backend.auth.orders.includes.print-pdf.stock';
                $data = [
                    'modal' => $order,
                    'user' => $user,
                    'profile' => $profile,
                    'stockOrder' => $stockOrder,
                ];
                break;
            case Order::ORDER_CONSUMABLE:
                $user = $order->user;
                $profile = $order->user->profile;
                $path = 'backend.auth.orders.includes.print-pdf.consumable';
                $data = [
                    'order' => $order,
                    'user' => $user,
                    'profile' => $profile
                ];
                break;
        }

        $pdf = PDF::loadView($path, $data);
        $pdfContent = $pdf->output();

        // Save to temp file
        $tempPath = storage_path('app/temp/' . $order->getOrderNumber() . '.pdf');
        if (!is_dir(storage_path('app/temp'))) {
            mkdir(storage_path('app/temp'), 0755, true);
        }
        file_put_contents($tempPath, $pdfContent);

        // Upload to Dropbox
        $dropboxPath = '/orders/' . $order->getOrderNumber() . '.pdf';
        try {
            DropboxComponent::uploadPdf($tempPath, $dropboxPath);
            $this->info('PDF uploaded to Dropbox successfully');
        } catch (\Exception $e) {
            $this->error('Failed to upload PDF: ' . $e->getMessage());
        }

        // Clean up temp file
        unlink($tempPath);
    }

    private function waitOrder($rawData)
    {
        $ordRes = [
            'brand' => $rawData['frame']['frame_info']['brand'] ?? '',
            'model' => $rawData['frame']['frame_info']['model'] ?? '',
            'color' => $rawData['frame']['frame_info']['colour'] ?? ''
        ];
        $rightData = ['sph' => floatval($rawData['rx']['distance']['right_sph'] ?? 0)];
        $leftData = ['sph' => floatval($rawData['rx']['distance']['left_sph'] ?? 0)];
        $maxPrismR = max(
            abs(floatval($rawData['rx']['prism']['right_prism1'] ?? 0)),
            abs(floatval($rawData['rx']['prism']['right_prism2'] ?? 0))
        );
        $maxPrismL = max(
            abs(floatval($rawData['rx']['prism']['left_prism1'] ?? 0)),
            abs(floatval($rawData['rx']['prism']['left_prism2'] ?? 0))
        );
        $frame_size_a = floatval($rawData['frame']['frame_size']['a'] ?? 0);
        $lensFilter = $rawData['lens']['lens_filter'] ?? '';
        $isTinted = (!empty($rawData['is_tinting']) && $rawData['is_tinting'] == 1) || stripos($lensFilter, 'tinted') !== false;
        $isPolar = (isset($rawData['tint_option']) && stripos($rawData['tint_option'], 'polarized') !== false) || stripos($lensFilter, 'polarized') !== false;

        $result = false;

        $frameDesc = ($ordRes['brand'] ?? '') . ' ' . ($ordRes['model'] ?? '') . ' ' . ($ordRes['color'] ?? '');

        if (($isPolar || $isTinted) && stripos($frameDesc, 'oakley') !== false) {
            $result = true;
        }

        if (stripos($frameDesc, 'lindberg') !== false || stripos($frameDesc, 'liberty') !== false) {
            $result = true;
        }

        if ($maxPrismR > 0.0 || $maxPrismL > 0.0) {
            $result = true;
        }

        if (($rightData['sph'] ?? 0) > 4.0 || ($leftData['sph'] ?? 0) > 4.0) {
            $result = true;
        }

        if (floatval($frame_size_a) >= 59.0) {
            $result = true;
        }

        return $result;
    }
}