<?php

namespace App\Http\Controllers\API;

use App\Services\LabOrderApiService;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Backend\Order;
use App\Models\Backend\RxOrders;
use App\Models\Backend\ConsumablesOrder;
use App\Models\Backend\StockOrder;

class LabOrderController extends Controller
{
    protected $labOrderService;

    private $jobTypeMapping = [
        'uncut' => 'UNCUT',
        'lab_supplies_frame' => 'SUPPLIED',
        'enclosed' => 'ENCLOSED',
        'edged_no_frame_to_follow' => 'LENSES ONLY',
        'edged_frame_to_follow' => 'TO COME'
    ];

    private $frameType = [
        'full_plastic' => 'PLASTIC',
        'full_metal' => 'METAL',
        'nyloon_groove' => 'GROOVE',
        'metal_groove' => 'MTLGRV',
        'rimless_drill' => 'DRILL'
    ];

    public function __construct(LabOrderApiService $labOrderService)
    {
        $this->labOrderService = $labOrderService;
    }

    /**
     * Check authentication token and return hash routing
     */
    public function checkAuthToken()
    {
        try {
            $token = $this->labOrderService->getAuthToken();
            if (!$token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Auth token is not available.'
                ], 500);
            }

            $hashRouting = $this->labOrderService->getHashRouting();
            $hashRoutingValue = $hashRouting['message']['lab']['contractSending'][0]['hash_routing'] ?? null;
            return response()->json([
                'success' => true,
                'token' => $token,
                'hash_routing' => $hashRoutingValue
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get auth token: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate common header array for all order types
     */
    public function generateHeaderArray($order, $orderType = null)
    {
        return [
            'file_version' => '2.6',
            'agent_version' => '2.10.26',
            'agent_name' => 'lens_shapers_web',
            'lab_num' => '1336',
            'cust_num' => $order->user->ac_code ?? '8888',
            'date_ordered' => date('Y-m-d-H-i-s'),
            'patient_name' => strtoupper($order->patient ?: ''),
            'customer_po_num' => ($order->method ?: '') . ($order->id ?: ''),
            'order_type' => $orderType ?: $order->type
        ];
    }

    /**
     * Generate RX content string from JSON data
     */
    public function generateRxContent($order, $jsonData)
    {
        $output = [];
        $output[] = "file_version:2.6";
        $output[] = "start_order";
        if (!empty($jsonData['header']['customer_po_num'])) {
            $output[] = "customer_po_num:" . $jsonData['header']['customer_po_num'];
        }
        $output[] = "agent_version:2.10.26";
        $output[] = "agent_name:lens_shapers_web";
        $output[] = "lab_num:1336";
        $output[] = "cust_num: " . ($order->user->ac_code ?? '8888');
        // $output[] = "cust_num: " . (($order->user->ac_code ?? '') . '8888');
        $output[] = "date_ordered:" . ($jsonData['header']['date_ordered'] ?? date('Y-m-d-H-i-s'));
        if (!empty($jsonData['header']['patient_name'])) {
            $output[] = "patient_name:" . $jsonData['header']['patient_name'];
        }
        
        $fields = $jsonData['items'][0]['fields'] ?? [];
        $fieldMap = [];
        foreach ($fields as $field) {
            $fieldMap[$field['labzilla_field']] = $field['value'];
        }

        if (!empty($fieldMap['frame_a'])) {
            $output[] = "frame_a:" . $fieldMap['frame_a'];
        }
        if (!empty($fieldMap['frame_b'])) {
            $output[] = "frame_b:" . $fieldMap['frame_b'];
        }
        if (!empty($fieldMap['frame_ed'])) {
            $output[] = "frame_ed:" . $fieldMap['frame_ed'];
        }
        if (!empty($fieldMap['frame_dbl'])) {
            $output[] = "frame_dbl:" . $fieldMap['frame_dbl'];
        }
        if (!empty($fieldMap['frame_vendor'])) {
            $output[] = "frame_vendor:" . $fieldMap['frame_vendor'];
        }
        if (!empty($fieldMap['frame_model'])) {
            $output[] = "frame_model:" . $fieldMap['frame_model'];
        }
        if (!empty($fieldMap['frame_color'])) {
            $output[] = "frame_color:" . $fieldMap['frame_color'];
        }
        if (!empty($fieldMap['frame_mounting'])) {
            $output[] = "frame_mounting:" . $fieldMap['frame_mounting'];
        }
        if (!empty($fieldMap['frame_status'])) {
            $output[] = "frame_status:" . $fieldMap['frame_status'];
        }

        $output[] = "lens_os_style_code:190";
        $output[] = "lens_os_material_code:0";
        $output[] = "lens_os_color_code:0";
        $output[] = "lens_os_vendor_code:83";
        
        $output[] = "lens_od_style_code:190";
        $output[] = "lens_od_material_code:0";
        $output[] = "lens_od_color_code:0";
        $output[] = "lens_od_vendor_code:83";
        
        $output[] = "rx_eye:3";
        if (!empty($fieldMap['rx_od_sphere'])) {
            $output[] = "rx_od_sphere:" . $fieldMap['rx_od_sphere'];
        }
        if (!empty($fieldMap['rx_od_cylinder'])) {
            $output[] = "rx_od_cylinder:" . $fieldMap['rx_od_cylinder'];
        }
        if (!empty($fieldMap['rx_od_axis'])) {
            $output[] = "rx_od_axis:" . $fieldMap['rx_od_axis'];
        }

        //IF (right_pd IS NOT NULL) THEN right_pd ELSEIF (right_npd IS NOT NULL) THEN right_npd ELSEIF (right_intermediate_pd IS NOT NULL) THEN right_intermediate_pd ELSE Do not send	rx_od_far
        $rx_od_far_val = null;
        if (!empty($fieldMap['rx_od_far'])) {

            $rx_od_far_val = $fieldMap['rx_od_far'];

        } elseif (!empty($fieldMap['rx_od_near'])) {

            $rx_od_far_val = $fieldMap['rx_od_near'];

        } elseif (!empty($fieldMap['rx_od_intermediate_pd'])) {
            
            $rx_od_far_val = $fieldMap['rx_od_intermediate_pd'];
        }

        if ($rx_od_far_val !== null) {
            $output[] = "rx_od_far:" . $rx_od_far_val;
        }

        //IF (right_pd IS NOT NULL AND right_npd IS NOT NULL) THEN right_npd ELSE Do Not Send	rx_od_near
        if (!empty($fieldMap['rx_od_far']) && !empty($fieldMap['rx_od_near'])) {
            $output[] = "rx_od_near:" . $fieldMap['rx_od_near'];
        }
        if (!empty($fieldMap['rx_od_seg_height'])) {
            $output[] = "rx_od_seg_height:" . $fieldMap['rx_od_seg_height'];
        }
        if (!empty($fieldMap['rx_od_prism'])) {
            $output[] = "rx_od_prism:" . $fieldMap['rx_od_prism'];
        }

        if (!empty($fieldMap['rx_os_sphere'])) {
            $output[] = "rx_os_sphere:" . $fieldMap['rx_os_sphere'];
        }
        if (!empty($fieldMap['rx_os_cylinder'])) {
            $output[] = "rx_os_cylinder:" . $fieldMap['rx_os_cylinder'];
        }
        if (!empty($fieldMap['rx_os_axis'])) {
            $output[] = "rx_os_axis:" . $fieldMap['rx_os_axis'];
        }


        //IF (left_pd IS NOT NULL) THEN left_pd ELSEIF (left_npd IS NOT NULL) THEN left_npd ELSEIF (left_intermediate_pd IS NOT NULL) THEN left_intermediate_pd ELSE Do not send	rx_os_far
        $rx_os_far_val = null;

        if (!empty($fieldMap['rx_os_far'])) {

            $rx_os_far_val = $fieldMap['rx_os_far'];

        } elseif (!empty($fieldMap['rx_os_near'])) {

            $rx_os_far_val = $fieldMap['rx_os_near'];

        } elseif (!empty($fieldMap['rx_os_intermediate_pd'])) {

            $rx_os_far_val = $fieldMap['rx_os_intermediate_pd'];
        }

        if ($rx_os_far_val !== null) {
            $output[] = "rx_os_far:" . $rx_os_far_val;
        }

        //IF (left_pd IS NOT NULL AND left_npd IS NOT NULL) THEN left_npd ELSE Do Not Send	rx_os_near
        if (!empty($fieldMap['rx_os_far']) && !empty($fieldMap['rx_os_near'])) {
            $output[] = "rx_os_near:" . $fieldMap['rx_os_near'];
        }
        if (!empty($fieldMap['rx_os_seg_height'])) {
            $output[] = "rx_os_seg_height:" . $fieldMap['rx_os_seg_height'];
        }
        if (!empty($fieldMap['rx_os_prism'])) {
            $output[] = "rx_os_prism:" . $fieldMap['rx_os_prism'];
        }
        if (!empty($fieldMap['instructions'])) {
            $output[] = "instructions:" . $fieldMap['instructions'];
        }
        if (!empty($fieldMap['instructions_2'])) {
            $output[] = "instructions_2:" . $fieldMap['instructions_2'];
        }
        if (!empty($fieldMap['instructions_3'])) {
            $output[] = "instructions_3:" . $fieldMap['instructions_3'];
        }
        if (!empty($fieldMap['instructions_4'])) {
            $output[] = "instructions_4:" . $fieldMap['instructions_4'];
        }
        if (!empty($fieldMap['instructions_5'])) {
            $output[] = "instructions_5:" . $fieldMap['instructions_5'];
        }

        $output[] = "end_order";
        return implode("\\r\\n", $output);
    }

     /**
     * Generate RX order array
     */
    public function rxOrderArray($order)
    {
        $rxOrder = RxOrders::where('order_id', $order->id)->first();
        if (!$rxOrder) {
            return response()->json([
                'success' => false,
                'message' => 'RX order details not found.'
            ], 404);
        }

        $showData = $rxOrder->getShowData();
        $fieldMappings = [
            ['our_field' => 'distance.right.sph', 'labzilla_field' => 'rx_od_sphere', 'label' => 'Right Sphere'],
            ['our_field' => 'distance.right.cyl', 'labzilla_field' => 'rx_od_cylinder', 'label' => 'Right Cylinder'],
            ['our_field' => 'distance.right.axis', 'labzilla_field' => 'rx_od_axis', 'label' => 'Right Axis'],
            ['our_field' => 'reading.right.add', 'labzilla_field' => 'rx_od_add', 'label' => 'Right Add'],

            ['our_field' => 'reading.right.fh', 'labzilla_field' => 'rx_od_seg_height', 'label' => 'Right Height'],
            ['our_field' => 'reading.right.pd', 'labzilla_field' => 'rx_od_far', 'label' => 'Right PD'],
            ['our_field' => 'reading.right.npd', 'labzilla_field' => 'rx_od_near', 'label' => 'Right NPD'],
            ['our_field' => 'reading.right.intermediate_pd', 'labzilla_field' => 'rx_od_intermediate_pd', 'label' => 'Right Intermediate PD'],

            ['our_field' => 'distance.right.prism1', 'labzilla_field' => 'rx_od_prism', 'label' => 'Right Prism'],


            ['our_field' => 'distance.left.sph', 'labzilla_field' => 'rx_os_sphere', 'label' => 'Left Sphere'],
            ['our_field' => 'distance.left.cyl', 'labzilla_field' => 'rx_os_cylinder', 'label' => 'Left Cylinder'],
            ['our_field' => 'distance.left.axis', 'labzilla_field' => 'rx_os_axis', 'label' => 'Left Axis'],
            ['our_field' => 'reading.left.add', 'labzilla_field' => 'rx_os_add', 'label' => 'Left Add'],


            ['our_field' => 'reading.left.fh', 'labzilla_field' => 'rx_os_seg_height', 'label' => 'Left Height'],
            ['our_field' => 'reading.left.pd', 'labzilla_field' => 'rx_os_far', 'label' => 'Left PD'],
            ['our_field' => 'reading.left.npd', 'labzilla_field' => 'rx_os_near', 'label' => 'Left NPD'],
            ['our_field' => 'reading.left.intermediate_pd', 'labzilla_field' => 'rx_os_intermediate_pd', 'label' => 'Left Intermediate PD'],


            ['our_field' => 'distance.left.prism1', 'labzilla_field' => 'rx_os_prism', 'label' => 'Left Prism'],

            ['our_field' => 'frame.frame_size.values.0', 'labzilla_field' => 'frame_a', 'label' => 'Frame A'],
            ['our_field' => 'frame.frame_size.values.1', 'labzilla_field' => 'frame_b', 'label' => 'Frame B'],
            ['our_field' => 'frame.frame_size.values.2', 'labzilla_field' => 'frame_ed', 'label' => 'Frame ED'],
            ['our_field' => 'frame.frame_size.values.3', 'labzilla_field' => 'frame_dbl', 'label' => 'Frame DBL'],
            ['our_field' => 'frame.job_type_key', 'labzilla_field' => 'frame_status', 'label' => 'Job Type'],
            ['our_field' => 'frame.frame_type_key', 'labzilla_field' => 'frame_mounting', 'label' => 'Frame Type'],
            ['our_field' => 'frame.frame_info.values.brand', 'labzilla_field' => 'frame_vendor', 'label' => 'Frame Brand'],
            ['our_field' => 'frame.frame_info.values.model', 'labzilla_field' => 'frame_model', 'label' => 'Frame Model'],
            ['our_field' => 'frame.frame_info.values.colour', 'labzilla_field' => 'frame_color', 'label' => 'Frame Color']
        ];

        $result = [
            'header' => $this->generateHeaderArray($order),
            'items' => []
        ];

        $item = [
            'item_start' => true,
            'rx_eye' => '3',
            'fields' => []
        ];

        foreach ($fieldMappings as $mapping) {
            $keys = explode('.', $mapping['our_field']);
            $value = $showData;
            foreach ($keys as $key) {
                if (isset($value[$key])) {
                    $value = $value[$key];
                } else {
                    $value = '';
                    break;
                }
            }

            if ($mapping['labzilla_field'] == 'frame_status') {
                $value = $this->jobTypeMapping[strtolower($value)] ?? 'UNCUT';
            }

            if ($mapping['labzilla_field'] == 'frame_mounting') {
                $value = $this->frameType[strtolower($value)] ?? '';
            }

            if ($value !== '' && $value !== null) {
                $item['fields'][] = [
                    'our_field' => $mapping['label'],
                    'labzilla_field' => $mapping['labzilla_field'],
                    'value' => $value
                ];
            }
        }

        $specialInstructions = $showData['additional_items']['special_instructions'] ?? '';
        if (!empty($specialInstructions)) {
            $chunks = str_split($specialInstructions, 185);
            foreach ($chunks as $index => $chunk) {
                if ($index < 5) {
                    $labzillaField = 'instructions' . ($index > 0 ? '_' . ($index + 1) : '');
                    $item['fields'][] = [
                        'our_field' => 'Special Instructions ' . ($index + 1),
                        'labzilla_field' => $labzillaField,
                        'value' => $chunk
                    ];
                }
            }
        }

        $item['fields'][] = [                                                             
                'our_field' => 'Frame Tracing',
                'labzilla_field' => 'frame_tracing',
                'value' => 'NO TRACE'
        ];

        $item['item_end'] = true;
        $result['items'][] = $item;

        $result['end_order'] = true;
        return $result;
    }

    /**
     * Generate consumable content string from JSON data
     */
    public function generateConsumableContent($order, $jsonData)
    {
        $output = [];
        $output[] = "file_version:2.6";
        $output[] = "start_order";
        $output[] = "customer_po_num:" . ($jsonData['header']['customer_po_num'] ?? '');
        $output[] = "agent_version:2.10.26";
        $output[] = "agent_name:lens_shapers_web";
        $output[] = "lab_num:1336";
        $output[] = "cust_num: " . ($order->user->ac_code ?? '8888');
        $output[] = "date_ordered:" . ($jsonData['header']['date_ordered'] ?? date('Y-m-d-H-i-s'));
        $output[] = "patient_name:" . ($jsonData['header']['patient_name'] ?? '');
        $output[] = "rx_eye:5";


        foreach ($jsonData['items'] as $item) {
            $output[] = "item_start";
            $output[] = "item_source:MISC";
            $output[] = "item_part_rx:N";
            foreach ($item['fields'] as $field) {
                $output[] = "{$field['labzilla_field']}:{$field['value']}";
            }
            $output[] = "item_end";
        }

        $output[] = "end_order";
        return implode("\\r\\n", $output);
    }

    /**
     * Generate stock content string from JSON data
     */
    public function generateStockContent($order, $jsonData)
    {
        $output = [];
        $output[] = "file_version:2.6";
        $output[] = "start_order";
        $output[] = "customer_po_num:" . ($jsonData['header']['customer_po_num'] ?? '');
        $output[] = "agent_version:2.10.26";
        $output[] = "agent_name:lens_shapers_web";
        $output[] = "lab_num:1336";
        $output[] = "cust_num: " . ($order->user->ac_code ?? '8888');
        $output[] = "date_ordered:" . ($jsonData['header']['date_ordered'] ?? date('Y-m-d-H-i-s'));
        $output[] = "patient_name:" . ($jsonData['header']['patient_name'] ?? '');
        $output[] = "rx_eye:5";
        
        foreach ($jsonData['items'] as $item) {
            $output[] = "item_start";
            $output[] = "item_source:FLENS";
            $output[] = "item_part_rx:N";
            foreach ($item['fields'] as $field) {
                $output[] = "{$field['labzilla_field']}:{$field['value']}";
            }
            $output[] = "item_end";
        }

        $output[] = "end_order";
        return implode("\\r\\n", $output);
    }

    /**
     * Generate stock order array
     */
    public function stockOrderArray($order)
    {
        $stockOrder = StockOrder::where('order_id', $order->id)->first();
        if (!$stockOrder) {
            return response()->json([
                'success' => false,
                'message' => 'Stock order not found.'
            ], 404);
        }

        $fieldMappings = [
            'barcode' => 'sku',
            'quantity' => 'item_quantity',
        ];

        $result = [
            'header' => $this->generateHeaderArray($order, Order::ORDER_STOCK),
            'items' => []
        ];

        $sides = [];
        if ($stockOrder->type === 'left_eye') {
            $sides = ['l'];
        } elseif ($stockOrder->type === 'right_eye') {
            $sides = ['r'];
        } else {
            $sides = ['l', 'r'];
        }

        foreach ($sides as $side) {
            $item = [
                'item_start' => true,
                'item_source' => 'FLENS',
                'item_part_rx' => 'N',
                'rx_eye' => '5',
                'fields' => []
            ];

            foreach ($fieldMappings as $ourField => $labzillaField) {
                $dbField = "{$side}_{$ourField}";
                $item['fields'][] = [
                    'our_field' => $dbField,
                    'labzilla_field' => $labzillaField,
                    'value' => $stockOrder->$dbField ?: ''
                ];
            }

            $item['item_end'] = true;
            $result['items'][] = $item;
        }

        $result['end_order'] = true;
        return $result;
    }

    /**
     * Generate consumable order array
     */
    public function consumableOrderArray($order)
    {
        $consumableOrders = ConsumablesOrder::where('order_id', $order->id)->get();
        if ($consumableOrders->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Consumable order details not found.'
            ], 404);
        }

        $fieldMappings = [
            'upc' => 'sku',
            'quantity' => 'item_quantity',
        ];

        $result = [
            'header' => $this->generateHeaderArray($order),
            'items' => []
        ];

        foreach ($consumableOrders as $consumableOrder) {
            $item = [
                'item_start' => true,
                'item_source' => 'MISC',
                'item_part_rx' => 'N',
                'rx_eye' => '5',
                'fields' => []
            ];

            foreach ($fieldMappings as $ourField => $labzillaField) {
                $item['fields'][] = [
                    'our_field' => $ourField,
                    'labzilla_field' => $labzillaField,
                    'value' => $consumableOrder->$ourField ?: ''
                ];
            }

            $item['item_end'] = true;
            $result['items'][] = $item;
        }

        $result['end_order'] = true;
        return $result;
    }

    /**
     * Placeholder for generating trace content
     */
    public function generateTrContent()
    {
        return "REQ=FIL\\r\\nJOB=?\\r\\nHBOX=51.00;51.00\\r\\nVBOX=38.00;38.00\\r\\nFED=51.00;51.00\\r\\nDBL=18\\r\\nTRCFMT=1;80;E;R;F\\r\\nR=2412;2469;2539;2613;2680;2729;2743;2698;2606;2495;2388;2293;2208;2134;2070\\r\\nR=2017;1974;1940;1917;1903;1900;1905;1920;1945;1979;2026;2083;2154;2237;2335\\r\\nR=2445;2566;2690;2763;2758;2713;2661;2618;2582;2555;2539;2538;2550;2578;2618\\r\\nR=2663;2698;2708;2685;2622;2514;2385;2268;2173;2098;2036;1989;1951;1923;1904\\r\\nR=1895;1895;1907;1928;1960;1999;2045;2096;2153;2208;2253;2279;2291;2294;2291\\r\\nR=2291;2294;2309;2332;2366";
    }


    /**
     * Fetch order data based on order type
     */
    public function getOrderData($orderId, Request $request)
    {
        try {
            $hashRouting = $request->input('hash_routing');
            $orderType = $request->input('order_type');
            $type = $request->input('type');
            $order = Order::findOrFail($orderId);
            $result = null;

            switch ($order->type) {
                case Order::ORDER_STOCK:
                    $result = $this->stockOrderArray($order);
                    break;
                case Order::ORDER_CONSUMABLE:
                    $result = $this->consumableOrderArray($order);
                    break;
                case Order::ORDER_RX:
                    $result = $this->rxOrderArray($order);
                    break;
                default:
                    return response()->json([
                        'success' => false,
                        'message' => 'Invalid order type.'
                    ], 400);
            }

            if ($result instanceof \Illuminate\Http\JsonResponse) {
                return $result;
            }

            if ($type == 'create') {
                $rxContent = null;
                if ($order->type === Order::ORDER_RX) {
                    $rxContent = $this->generateRxContent($order, $result);
                } elseif ($order->type === Order::ORDER_CONSUMABLE) {
                    $rxContent = $this->generateConsumableContent($order, $result);
                } elseif ($order->type === Order::ORDER_STOCK) {
                    $rxContent = $this->generateStockContent($order, $result);
                }
                $trContent = $this->generateTrContent();
                return response()->json([
                    'success' => true,
                    'data' => [
                        'order' => [
                            'hash_routing' => $hashRouting,
                            'rx_content' => $rxContent,
                            'tr_content' => $trContent
                        ]
                    ]
                ]);
            } else {
                return response()->json([
                    'success' => true,
                    'data' => $result,
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch order data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Push order to LabZilla API
     */
    public function pushOrder(Request $request)
    {

        try {
            $validatedData = $request->validate([
                'orderData.order.hash_routing' => 'required|string',
                'orderData.order.rx_content' => 'required|string',
                'orderData.order.tr_content' => 'nullable|string',
                'orderId' => 'required|exists:orders,id',
                'manualClick' => 'nullable|boolean'
            ]);

            $order = Order::findOrFail($validatedData['orderId']);
            $orderData = $validatedData['orderData']['order'];
            $manualClick = $validatedData['manualClick'] ?? false;

            $response = $this->labOrderService->pushOrderToLab($orderData, $order, $manualClick);
            
            if (isset($response['success']) && !$response['success']) {
                return response()->json($response, 400);
            }

            return response()->json([
                'success' => true,
                'data' => $response
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to push order: ' . $e->getMessage()
            ], 500);
        }
    }
}