<?php

namespace App\Http\Controllers\Backend\Auth;

use Carbon\Carbon;

use Illuminate\Http\Request;
use App\Models\Backend\Order;
use App\Models\Backend\RxOrders;
use App\Models\Backend\AppSettings;
use App\Models\Backend\PhoneLog;
use App\Models\Backend\StockOrder;
use App\Models\Backend\OrderStatus;
use App\Http\Controllers\Controller;
use App\Models\Backend\RxValidations;
use App\Models\Backend\EmailTemplate;
use App\Models\Backend\Notification;
use App\Http\Controllers\Backend\Auth;
use App\Models\Backend\ConsumablesOrder;
use App\Custom\DropboxComponent;
use App\Events\Frontend\Auth\CustomerInquiryLog;
use App\Models\Backend\ImportedConsumablesOrder;
use App\Models\Backend\Stockvalidation;
use App\Models\Backend\ScheduledCrons;
use App\Models\Backend\FrameStyles;
use App\Services\CommentService;
use App\Models\Backend\LabOrders;
use App\Http\Controllers\CommonController;
use PDF;
use Illuminate\Support\Facades\Storage;

class OrdersController extends Controller
{

    protected $commentService;

    public function __construct(CommentService $commentService)
    {
        $this->commentService = $commentService;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $currentPage    = $request->get('page-no', 1);
        $recordsPerPage = $request->get('records-per-page', config('access.list_page_size'));
        $orders = [];
        $models = OrderStatus::all();
        $statuses = $models->map(function ($status) {
            return [
                'id' => $status->id,
                'text' => $status->title
            ];
        })->values()->toArray();
        if($request->ajax()){
            $order = new Order();
            $orders = $order->search($request, $currentPage, $recordsPerPage);
            $totalRecords = $orders->total();
            $loadedRecords = $orders->count();
            return response()->json([
                'html' => view('partials.orderlist', [
                    'isBackend' => true,
                    'orders' => $orders,
                    'pageCount' => $currentPage,
                    'RecordsPerPage' => $recordsPerPage,
                    'orderType' => 'all'
                ])->render(),
                'totalRecords' => $totalRecords,
                'loadedRecords' => $loadedRecords
            ]);
        }

        // Get total records for initial load
        $orderModel = new Order();
        $tempOrders = $orderModel->search($request, 1, 1); // Get paginated to get total
        $totalRecords = $tempOrders->total();

        $customers = \App\Models\Auth\User::getUserByRoleName(config('access.users.customer_admin'))->sortBy('ac_code');
        $customerData = $customers->map(function ($customer) {
            return [
                'id' => $customer->id,
                'text' => $customer->ac_code . ' - ' . $customer->business_name
            ];
        })->values()->toArray();

        return view('backend.auth.orders.index')
            ->withOrders($orders)
            ->withModels($models)
            ->withStatuses($statuses)
            ->withPageCount($currentPage)
            ->withRecordsPerPage($recordsPerPage)
            ->withTemplates(EmailTemplate::all())
            ->withOrderType('all')
            ->withTotalRecords($totalRecords)
            ->withCustomers($customerData);

    }

    public function list(Request $request)
    {
        $currentPage    = $request->get('page-no', 1);
        $recordsPerPage = $request->get('records-per-page', config('access.list_page_size'));
        $orderId        = $request->get('order', 0);
        $models         = OrderStatus::all();
        $orders         = [];
        $order = new Order();
        $orders = $order->search($request, $currentPage, $recordsPerPage);
        if ($request->ajax()) {
            // Get paginated results with custom page size
                $totalRecords = $orders->total();
                $loadedRecords = $orders->count();
                return response()->json([
                    'html' => view('partials.orderlist', [
                        'isBackend' => true,
                        'orders' => $orders,
                        'pageCount' => $currentPage,
                        'RecordsPerPage' => $recordsPerPage,
                        'OrderId' => $orderId,
                        'orderType' => 'all'
                    ])->render(),
                    'totalRecords' => $totalRecords,
                    'loadedRecords' => $loadedRecords
                ]);
        }

        $statuses = OrderStatus::all();
        $statusData = $statuses->map(function ($status) {
            return [
                'id' => $status->id,
                'text' => $status->title
            ];
        })->values()->toArray();

        $customers = \App\Models\Auth\User::getUserByRoleName(config('access.users.customer_admin'))->sortBy('ac_code');
        $customerData = $customers->map(function ($customer) {
            return [
                'id' => $customer->id,
                'text' => $customer->ac_code . ' - ' . $customer->business_name
            ];
        })->values()->toArray();

        return app(CommonController::class)->renderOrdersView([
            'layout' => 'backend.layouts.split',
            'title' => 'Orders' . ' '.html_entity_decode("&#8212;", ENT_QUOTES, 'UTF-8').' ' . app_name(),
            'isBackend' => true,
            'searchFormInclude' => 'backend.auth.orders.includes.orders-search-form',
            'pageType' => null,
            'orders' => $orders,
            'pageCount' => $recordsPerPage,
            'orderId' => $orderId,
            'orderData' => null,
            'commentsData' => null,
            'templates' => \App\Models\Backend\EmailTemplate::all(),
            'statuses' => $statusData,
            'customers' => $customerData,
        ]);
    }



    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function rxLensOrders(Request $request)
    {
        $pageCount = config('access.list_page_size');
        if(!empty($request->get('page-no')))
            $pageCount = $request->get('page-no') + $pageCount;

        $orders = Order::orderBy('created_at','desc')->where('type', Order::ORDER_RX)->paginate($pageCount);
        $models = OrderStatus::all();
        $statuses = $models->map(function ($status) {
            return [
                'id' => $status->id,
                'text' => $status->title
            ];
        })->values()->toArray();

        if($request->ajax()){
                return view('partials.orderlist', ['isBackend' => true])
                    ->with('orders', $orders)
                    ->with('pageCount', $pageCount)
                    ->with('orderType', Order::ORDER_RX);
        }

        return app(CommonController::class)->renderOrdersView([
            'layout' => 'backend.layouts.split',
            'title' => 'Orders' . ' '.html_entity_decode("&#8212;", ENT_QUOTES, 'UTF-8').' ' . app_name(),
            'isBackend' => true,
            'searchFormInclude' => 'backend.auth.orders.includes.orders-search-form',
            'pageType' => null,
            'orders' => $orders,
            'pageCount' => $pageCount,
            'orderId' => null,
            'orderData' => null,
            'commentsData' => null,
            'templates' => \App\Models\Backend\EmailTemplate::all(),
            'statuses' => $statuses,
            'customers' => [],
        ]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function stockLensOrders(Request $request)
    {
        $pageCount = config('access.list_page_size');
        if(!empty($request->get('page-no')))
            $pageCount = $request->get('page-no') + $pageCount;

        $orders = Order::orderBy('created_at','desc')->where('type', Order::ORDER_STOCK)->paginate($pageCount);
        $models = OrderStatus::all();
        $statuses = $models->map(function ($status) {
            return [
                'id' => $status->id,
                'text' => $status->title
            ];
        })->values()->toArray();
        if($request->ajax()){

            return view('backend.auth.orders.includes.stock-table')
                ->withOrders($orders)
                ->withPageCount($pageCount)
                ->withOrderType(Order::ORDER_STOCK);
        }

        return view('backend.auth.orders.stock-orders')
            ->withOrders($orders)
            ->withModels($models)
            ->withStatuses($statuses)
            ->withPageCount($pageCount)
            ->withTemplates(EmailTemplate::all())
            ->withOrderType(Order::ORDER_STOCK);
    }
    /**
     * Search Barcode Api.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function searchBarcodeApi(Request $request)
    {
        if($request->ajax() && !empty($request->get('value'))){
            $data = Stockvalidation::where('barcode',str_replace(' ', '', $request->get('value')))->first();
            return response()->json(['status'=>true,'data'=>$data]);
        }
        return response()->json(['status'=>false]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function consumableLensOrders(Request $request)
    {
        $pageCount = config('access.list_page_size');
        if(!empty($request->get('page-no')))
            $pageCount = $request->get('page-no') + $pageCount;

        $orders = Order::orderBy('created_at','desc')->where('type', Order::ORDER_CONSUMABLE)->paginate($pageCount);
        $models = OrderStatus::all();
        $statuses = $models->map(function ($status) {
            return [
                'id' => $status->id,
                'text' => $status->title
            ];
        })->values()->toArray();

        if($request->ajax()){
            return view('backend.auth.orders.includes.consumable-table')
                ->withOrders($orders)
                ->withPageCount($pageCount)
                ->withOrderType(Order::ORDER_CONSUMABLE);
        }

        return view('backend.auth.orders.consumable-orders')
            ->withOrders($orders)
            ->withModels($models)
            ->withStatuses($statuses)
            ->withPageCount($pageCount)
            ->withTemplates(EmailTemplate::all())
            ->withOrderType(Order::ORDER_CONSUMABLE);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function barcodeLensOrders()
    {
        return view('backend.auth.orders.barcode-orders');
    }

    public function searchOrders(Order $order, Request $request)
    {
        $currentPage    = $request->get('page-no', 1);
        $recordsPerPage = $request->get('records-per-page', config('access.list_page_size'));
        if ($request->ajax()) {
            $orders = $order->search($request, $currentPage, $recordsPerPage);
            $totalRecords = $orders->total();
            $loadedRecords = $orders->count();
            // return view('partials.orderlist', ['isBackend' => true])
            //         ->with('orders', $orders)
            //         ->with('pageCount', $currentPage)
            //         ->with('RecordsPerPage', $recordsPerPage)
            //         ->with('orderType', $request->get('order_type'));
            return response()->json([
                'html' => view('partials.orderlist', [
                    'isBackend' => true,
                    'orders' => $orders,
                    'pageCount' => $currentPage,
                    'RecordsPerPage' => $recordsPerPage,
                    'orderType' => $request->get('order_type')
                ])->render(),
                'totalRecords' => $totalRecords,
                'loadedRecords' => $loadedRecords
            ]);
        }

      
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function show(Order $order) {
        switch ($order->type) {
            case Order::ORDER_STOCK:
                $stockOrder = StockOrder::where('order_id', $order->id)->first();
                return view('backend.auth.orders.includes.show.stock')
                    ->withStockOrder($stockOrder)
                    ->withModal($order);

                break;
            case Order::ORDER_CONSUMABLE:
                return view('backend.auth.orders.includes.show.consumable')
                    ->withModal($order);

            case Order::ORDER_RX:
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                $rxOrderArray   = $rxOrder->getShowData();
                return view('backend.auth.orders.includes.show.rx')
                    ->withModal($order)
                    ->withRxOrder($rxOrder)
                    ->withRxOrderArray($rxOrderArray);
                break;
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function print(Request $request, Order $order)
    {
        switch ($order->type) {
            case Order::ORDER_RX:
                $user = $order->user;
                $profile = $order->user->profile;
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                $rxOrderArray   = $rxOrder->getShowData();
                $rawData = json_decode($rxOrder->data, true);
                $shouldWait = $this->waitOrder($rawData);
                return view('backend.auth.orders.includes.print.rx')
                    ->withOrder($order)
                    ->withRxOrder($rxOrder)
                    ->withUser($user)
                    ->withProfile($profile)
                    ->withRxOrderArray($rxOrderArray)
                    ->withShouldWait($shouldWait);
                break;
            case Order::ORDER_BARCODE: case Order::ORDER_STOCK:
                $stockOrder = StockOrder::where('order_id', $order->id)->first();
                $user = $order->user;
                $profile = $order->user->profile;
                return view('backend.auth.orders.includes.print.stock')
                    ->withModal($order)
                    ->withUser($user)
                    ->withProfile($profile)
                    ->withStockOrder($stockOrder);
                break;
            case Order::ORDER_CONSUMABLE:
                $user = $order->user;
                $profile = $order->user->profile;
                return view('backend.auth.orders.includes.print.consumable')
                    ->withOrder($order)
                    ->withUser($user)
                    ->withProfile($profile);
                break;
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Order  $order
     * @return \Illuminate\Http\Response
     */
    public function printPdf(Request $request, Order $order)
    {
        $html = '';
        switch ($order->type) {
            case Order::ORDER_RX:
                $user = $order->user;
                $profile = $order->user->profile;
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                $rxOrderArray   = $rxOrder->getShowData();
                $rawData = json_decode($rxOrder->data, true);
                $shouldWait = $this->waitOrder($rawData);
                $path = 'backend.auth.orders.includes.print-pdf.rx';
                $data = [
                    'order' =>  $order,
                    'rxOrder' =>  $rxOrder,
                    'user' =>  $user,
                    'profile' =>  $profile,
                    'rxOrderArray' =>  $rxOrderArray,
                    'shouldWait' => $shouldWait,
                ];
                break;
            case Order::ORDER_BARCODE: case Order::ORDER_STOCK:
                $stockOrder = StockOrder::where('order_id', $order->id)->first();
                $user = $order->user;
                $profile = $order->user->profile;
                // $html = view('backend.auth.orders.includes.print-pdf.stock')
                //     ->withModal($order)
                //     ->withUser($user)
                //     ->withProfile($profile)
                //     ->withStockOrder($stockOrder)->render();
                $path = 'backend.auth.orders.includes.print-pdf.stock';
                $data = [
                    'modal' =>  $order,
                    'user' =>  $user,
                    'profile' =>  $profile,
                    'stockOrder' =>  $stockOrder,
                ];
                break;
            case Order::ORDER_CONSUMABLE:
                $user = $order->user;
                $profile = $order->user->profile;
                // $html = view('backend.auth.orders.includes.print-pdf.consumable')
                //     ->withOrder($order)
                //     ->withUser($user)
                //     ->withProfile($profile)->render();
                $path = 'backend.auth.orders.includes.print-pdf.consumable';
                $data = [
                    'order' =>  $order,
                    'user' =>  $user,
                    'profile' =>  $profile
                ];
                break;
        }
        $pdf = PDF::loadView($path, $data);

        // return $pdf->download($order->getOrderNumber().'.pdf');
        // return $pdf->stream($order->getOrderNumber().'.pdf',array('Attachment'=>0));
        return $pdf->stream('LS-Invoice-' .$order->getOrderNumber().'.pdf',['Attachment'=>0]);
    }
    /**
     *
     * @return \Illuminate\Http\Response
     */
    public function settings(Request $request)
    {
        if($request->ajax() && $request->post()){
            $metaData = [];
            $rules = [
                'type' => ['required'],
                // 'attachment_file'  =>  ['required','mimes:csv']
            ];
            $validator = \Validator::make($request->all(), $rules);
            if($validator->fails()){
                return response()->json(['status'=>false,'message' => implode(",",$validator->messages()->all())]);
            }
            $status = false;
            $message = __('Something Went Wrong');
            $flag = false;
            $showAlert = false;
            switch ($request->get('type')) {
                case "rx":
                    if($request->file('attachment_file')->storeAS('public/imports','lens_data.csv')){
                        // $flag = RxValidations::import();
                        $status = true;
                        $showAlert = true;
                        $message = __('Rx data importing in progress. Please Wait!');
                        ScheduledCrons::add('rx','Rx data importing in progress. Please Wait!');
                    }
                break;
                case "stock":
                    if($request->file('attachment_file')->storeAS('public/imports','stock_lenses.csv'))
                        $flag = Stockvalidation::import();
                break;
                case "consumable":
                    if($request->file('attachment_file')->storeAS('public/imports',ImportedConsumablesOrder::FILE_NAME)){
                        $flag = ImportedConsumablesOrder::import();
                        if($flag)
                            $metaData['images'] = ImportedConsumablesOrder::uploadImages($request->file('images'));
                    }
                break;
                case "consumable-images":
                    if($request->file('file')){
                        $metaData['images'] = ImportedConsumablesOrder::uploadImage($request->file('file'));
                        $status = true;
                        $message = __('File uploaded successfully.');
                    }
                break;
                case "consumable-image-remove":
                    if($request->get('filename')){
                        ImportedConsumablesOrder::removeImage($request->get('filename'));
                        $status = true;
                        $message = __('File removed successfully.');
                    }
                break;
                case "rx_lens_addons":
                    if($request->file('attachment_file')->storeAS('public/imports','rx_lens_addons.csv'))
                        $flag = RxValidations::importRxAddons();
                break;
                case "rx_additional_charges":
                    if($request->file('attachment_file')->storeAS('public/imports','rx_additional_charges.csv'))
                        $flag = RxValidations::importRxAdditionalCharges();
                break;
                case "rx_base_cruve":
                    if($request->file('attachment_file')->storeAS('public/imports','base-curve-calculations.csv'))
                        $flag = RxValidations::importRxBaseCurve();
                break;
                case "assets-images":
                    if($request->file('file')){
                        $metaData['images'] = RxOrders::uploadAssetsImage($request->file('file'));
                        $status = true;
                        $message = __('File uploaded successfully.');
                    }
                break;
                case "assets-remove":
                    if($request->get('filename')){
                        RxOrders::removeAssetsImage($request->get('filename'));
                        $status = true;
                        $message = __('File removed successfully.');
                    }
                break;
                case "flyers-images":
                    if($request->file('file')){
                        $metaData['images'] = AppSettings::uploadFlyersImage($request->file('file'));
                        $status = true;
                        $message = __('File uploaded successfully.');
                    }
                break;
                case "dashboard_content":
                    if($request->get('dashboard_content2')){
                        $obj = AppSettings::findByObj(AppSettings::DEFAULT_DASHBOARD_CONTENT);
                        if(!empty($obj)){
                            $obj->settings_value = $request->get('dashboard_content2');
                            $obj->update();
                        }
                    }
                    $status = true;
                    $message = __('Data updated successfully');
                break;
                case "flyers-remove":
                    if($request->get('filename')){
                        AppSettings::removeFlyersImage($request->get('filename'));
                        $status = true;
                        $message = __('File removed successfully.');
                    }
                break;
                case "stock_center_price":
                    $status = false;
                    $message = __('Something went wrong. Please try again later!');
                    if($request->file('attachment_file')->storeAS('public/uploads',AppSettings::STOCK_CENTER_PRICE_FILE)){
                        $status = true;
                        $message = __('File updated successfully');
                    }
                break;
                case "help_video_link":
                    $status = true;
                    $message = __('Url saved successfully');
                    $model = AppSettings::findByObj(AppSettings::DEFAULT_HOW_TO_START_VIDEO);
                    $model->settings_value = !empty($request->url) ? $request->url : '';
                    $model->update();
                break;
                case "assets-model-images":
                    if($request->file('file')){
                        $uploaded = FrameStyles::uploadAssetsImage($request->file('file'));
                        if(!empty($uploaded)){
                            $metaData['images'] = $uploaded;
                            $status = true;
                            $message = __('File uploaded successfully.');
                        } else {
                            $status = false;
                            $message = __('File upload failed.');
                        }
                    } else {
                        $status = false;
                        $message = __('No file provided.');
                    }
                break;
                case "assets-model-remove":
                    if($request->get('filename')){
                        FrameStyles::removeAssetsImage($request->get('filename'));
                        $status = true;
                        $message = __('File removed successfully.');
                    }
                break;
            }
            if($flag){
                $status = true;
                $message = __('Data has been updated.');
            }
            return response()->json(['status'=>$status,'message' => $message,'metaData'=>$metaData,'show_alert'=>$showAlert]);
        }
        $allImages = ConsumablesOrder::getAllImages();
        $allAssetsImages = RxOrders::getAllImages();
        $allFlyersImages = AppSettings::getAllFlyerImages();
        $modelAssetsImages = FrameStyles::getModelImages();
        $dashboardContent = AppSettings::findByKey(AppSettings::DEFAULT_DASHBOARD_CONTENT);
        $labOrders = new LabOrders();
        $usersForSelect = $labOrders->getUsersForSelect();
        $orderType = $labOrders->getOrderType();
        $isEnable  = $labOrders->isEnable();
        $savedConfig = getLabzillaAccountConfig();
        return view('backend.auth.orders.settings',['allImages'=>$allImages,'allAssetsImages'=>$allAssetsImages,'allFlyersImages'=>$allFlyersImages,'dashboardContent'=>$dashboardContent, 'modelAssetsImages' =>$modelAssetsImages, 'labOrders' => $labOrders, 'usersForSelect' => $usersForSelect, 'savedConfig' => $savedConfig, 'orderType' => $orderType, 'isEnable' => $isEnable]);
    }

    /**
     * Print to dropbox.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function uploadPrint(Request $request,$id)
    {
        if(!empty($id)){
            // TODO: Check if PDF already exists in Dropbox for this order
            // If exists, skip creation, else proceed
            \Artisan::call('upload:dropboxfiles', ['id' => $id]);
            if($request->ajax()){
                return response()->json(['status'=>true,'message'=>'Print sent successfully']);
            } else {
                return redirect()->back()->with('success', 'Print sent to Dropbox');
            }
        }
        if($request->ajax()){
            return response()->json(['status'=>false,'message'=>'Something went wrong']);
        } else {
            return redirect()->back()->with('error', 'Something went wrong');
        }
    }

    public function detailsTypeApi(Request $request)
    {
        return $this->commentService->handleRequest($request, true);
    }

    public function saveLabOrderConfiguration(Request $request)
    {
        $userAccounts = $request->input('user_accounts', []);
        
        // If 'all' is in user_accounts, only save 'all'
        if(!empty($userAccounts)) {
            if (in_array('all', $userAccounts)) {
                $userAccounts = ['all'];
            }
        }

        $data = [
            'user_accounts' => $userAccounts,
            'order_types' => $request->input('order_types', []),
            'status' => $request->input('status', false),
        ];

        $jsonData = json_encode($data, JSON_PRETTY_PRINT);
        Storage::put('labzilla-account.txt', $jsonData);
        return response()->json(['success' => true, 'message' => 'Configuration saved.']);
    }

    private function waitOrder($rawData) {
        $ordRes = [
            'brand' => $rawData['frame']['frame_info']['brand'] ?? '',
            'model' => $rawData['frame']['frame_info']['model'] ?? '',
            'color' => $rawData['frame']['frame_info']['colour'] ?? ''
        ];
        $rightData = ['sph' => floatval($rawData['rx']['distance']['right_sph'] ?? 0)];
        $leftData = ['sph' => floatval($rawData['rx']['distance']['left_sph'] ?? 0)];
        $maxPrismR = max(
            abs(floatval($rawData['rx']['prism']['right_prism1'] ?? 0)),
            abs(floatval($rawData['rx']['prism']['right_prism2'] ?? 0))
        );
        $maxPrismL = max(
            abs(floatval($rawData['rx']['prism']['left_prism1'] ?? 0)),
            abs(floatval($rawData['rx']['prism']['left_prism2'] ?? 0))
        );
        $frame_size_a = floatval($rawData['frame']['frame_size']['a'] ?? 0);
        $lensFilter = $rawData['lens']['lens_filter'] ?? '';
        $isTinted = (!empty($rawData['is_tinting']) && $rawData['is_tinting'] == 1) || stripos($lensFilter, 'tinted') !== false;
        $isPolar = (isset($rawData['tint_option']) && stripos($rawData['tint_option'], 'polarized') !== false) || stripos($lensFilter, 'polarized') !== false;

        $result = false;

        $frameDesc = ($ordRes['brand'] ?? '') . ' ' . ($ordRes['model'] ?? '') . ' ' . ($ordRes['color'] ?? '');

        if (($isPolar || $isTinted) && stripos($frameDesc, 'oakley') !== false) {
            $result = true;
        }

        if (stripos($frameDesc, 'lindberg') !== false || stripos($frameDesc, 'liberty') !== false) {
            $result = true;
        }

        if ($maxPrismR > 0.0 || $maxPrismL > 0.0) {
            $result = true;
        }

        if (($rightData['sph'] ?? 0) > 4.0 || ($leftData['sph'] ?? 0) > 4.0) {
            $result = true;
        }

        if (floatval($frame_size_a) >= 59.0) {
            $result = true;
        }

        return $result;
    }
}
