<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use App\Models\Backend\Order;
use App\Models\Backend\RxOrders;
use App\Models\Backend\StockOrder;
use App\Models\Backend\PhoneLog;
use App\Models\Backend\Notification;
use App\Models\Backend\AppSettings;

/**
 * Class CommonController.
 */
class CommonController extends Controller
{
    public function detailsApi(Order $order): JsonResponse
    {
        if (!$order->exists) {
            return response()->json(['error' => 'Order not found'], 404);
        }

        $user = auth()->user();
        $isAdmin = $user->hasRole('administrator') || $user->can('view backend');
        $customerAdminId = $isAdmin ? 1 : (function () use ($user) {
            try {
                $customerAdmin = $user->getCustomerAdmin();
                return $customerAdmin ? $customerAdmin->id : $user->id;
            } catch (\Exception $e) {
                return $user->id;
            }
        })();

        $response = [
            'order' => '',
            'comments' => '',
            'invoice' => '',
            'notifications_count' => Notification::where('order_id', $order->id)
                ->where(function ($query) use ($customerAdminId, $user, $isAdmin) {
                    $query->where('to', $customerAdminId);
                    if (!$isAdmin) {
                        $query->orWhere('to', $user->id);
                    }
                })
                ->where('has_read', 0)
                ->count(),
        ];

        $response['order'] = match ($order->type) {
            Order::ORDER_STOCK => view('partials.stock', [
                'order' => $order,
                'stockOrder' => StockOrder::where('order_id', $order->id)->first(),
            ])->render(),
            Order::ORDER_CONSUMABLE => view('partials.consumable', [
                'order' => $order,
            ])->render(),
            Order::ORDER_RX => (function () use ($order) {
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                return $rxOrder ? view('partials.rx', [
                    'order' => $order,
                    'rxOrder' => $rxOrder,
                    'rxOrderArray' => $rxOrder->getShowData(),
                ])->render() : '';
            })(),
            default => '',
        };

        $logs = PhoneLog::with(['replies' => function($query) {
            $query->with('user')->orderBy('created_at', 'asc');
        }])
        ->where('order_id', $order->id)
        ->whereNull('replied')
        ->when(!$isAdmin, function($query) {
            $query->where('type', '!=', 'notes');
        })
        ->orderBy('created_at', 'desc')
        ->get();

        $orderType = $order->type;

        $rxOrder =  $rxOrderArray =  $stockOrder = null;

        // Only get RxOrder data if order type is 'rx'
        if ($orderType === 'rx') {
            $rxOrder = RxOrders::where('order_id', $order->id)->first();
            $rxOrderArray = $rxOrder ? $rxOrder->getShowData() : null;
        }

        // Get stock order if needed
        $stockOrder = StockOrder::where('order_id', $order->id)->first();

        $response['comments'] = view('partials.chatbot_grid', [
            'logs' => $logs,
            'user' => $user,
            'sender' => $isAdmin ? 'admin': 'customer',
            'showButtons' => false,
            'isCustomerAdmin' => $user->isCustomerAdmin(),
            'type' => $orderType,
            'order' => $order,
            'stockOrder' => $stockOrder,
            'rxOrder' => $rxOrder,
            'rxOrderArray' => $rxOrderArray,
        ])->render();

        return response()->json($response);
    }

    public function dashboard(): View|RedirectResponse
    {
        $isCustomerAdmin = auth()->user()->hasRole('customer_admin');
        $isCustomerStaff = auth()->user()->hasRole('customer_staff');
        if ($isCustomerAdmin || $isCustomerStaff) {
            $dashboardContent = AppSettings::findByKey(AppSettings::DEFAULT_DASHBOARD_CONTENT);
            return view('frontend.user.dashboard')
                ->withDashboardContent($dashboardContent);
        } else {
            return redirect()->route('admin.orders.list');
        }
    }

    public function renderOrdersView(array $data): View
    {
        $content = view('partials.orders', $data);
        return view($data['layout'], $data)->with('content', $content);
    }
}