<?php

namespace App\Http\Controllers\Frontend\Auth;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use App\Models\Backend\Order;
use App\Models\Backend\RxOrders;
use App\Models\Backend\StockOrder;
use App\Models\Backend\OrderStatus;
use App\Http\Controllers\Controller;
use App\Models\Backend\Stockvalidation;
use App\Models\Backend\ConsumablesOrder;
use Illuminate\Support\Facades\Validator;
use App\Models\Backend\ImportedConsumablesOrder;
use App\Models\Backend\RxValidations;
use App\Http\Requests\Frontend\Auth\StoreRxOrderRequest;
use App\Http\Requests\Frontend\Auth\StoreStockOrderRequest;
use App\Http\Requests\Frontend\Auth\ManageStockOrderRequest;
use App\Http\Requests\Frontend\Auth\ManageCreateOrderRequest;
use App\Models\Backend\PhoneLog;
use App\Models\Backend\Notification;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use App\Services\CommentService;
use App\Http\Controllers\CommonController;

/**
 * Class OrdersController.
 */
class OrdersController extends Controller
{
    protected CommentService $commentService;

    public function __construct(CommentService $commentService)
    {
        $this->commentService = $commentService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return View|JsonResponse
     */
    public function index(Request $request): View|JsonResponse
    {
        $currentPage = $request->get('page-no', 1);
        $recordsPerPage = $request->get('records-per-page', config('access.list_page_size'));
        $orderId = $request->get('order', 0);
        $orders = [];
        $order = new Order();
        $orders = $order->search($request, $currentPage, $recordsPerPage);

        if ($orderId) {
            $orderModel = Order::find($orderId);
            if ($orderModel && $orderModel->user_id == auth()->user()->getCustomerAdmin()->id) {
                $ordersArray = $orders->getCollection();
                if (!$ordersArray->contains('id', $orderId)) {
                    $ordersArray->prepend($orderModel);
                    $orders = new \Illuminate\Pagination\LengthAwarePaginator($ordersArray, $orders->total() + 1, $recordsPerPage, $currentPage);
                }
                // Load order details and comments only if not from notification
                if (!$request->has('from') || $request->get('from') !== 'notification') {
                    $orderData = $this->getOrderDetails($orderId);
                    $commentsData = $this->getOrderComments($orderId);
                }
            }
        }

        $statuses = OrderStatus::getCustomerOrderStatus();
        $statusData = collect($statuses)->map(fn($status) => [
            'id' => $status[0]['id'] ?? null,
            'text' => $status[0]['title'] ?? ''
        ])->filter()->values()->all();

        if ($request->ajax()) {
            if ($orderId) {
                $orderModel = Order::find($orderId);
                if ($orderModel && $orderModel->user_id == auth()->user()->getCustomerAdmin()->id) {
                    $ordersArray = $orders->getCollection();
                    if (!$ordersArray->contains('id', $orderId)) {
                        $ordersArray->prepend($orderModel);
                        $orders = new \Illuminate\Pagination\LengthAwarePaginator($ordersArray, $orders->total() + 1, $recordsPerPage, $currentPage);
                    }
                }
            }
            $totalRecords = $orders->total();
            $loadedRecords = $orders->count();
            return response()->json([
                'html' => view('partials.orderlist', [
                    'isBackend' => false,
                    'orders' => $orders,
                    'pageCount' => $currentPage,
                    'RecordsPerPage' => $recordsPerPage,
                    'OrderId' => $orderId,
                    'orderType' => 'all'
                ])->render(),
                'totalRecords' => $totalRecords,
                'loadedRecords' => $loadedRecords
            ]);
        }

        return app(CommonController::class)->renderOrdersView([
            'layout' => 'frontend.layouts.split',
            'title' => 'Orders' . ' '.html_entity_decode("&#8212;", ENT_QUOTES, 'UTF-8').' ' . app_name(),
            'isBackend' => false,
            'searchFormInclude' => 'frontend.auth.orders.includes.orders-search-form',
            'pageType' => 0,
            'orders' => $orders,
            'pageCount' => $recordsPerPage,
            'orderId' => $orderId,
            'orderData' => $orderData ?? null,
            'commentsData' => $commentsData ?? null,
            'statuses' => $statusData,
        ]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return View|JsonResponse
     */
    public function saved(Request $request): View|JsonResponse
    {
        $currentPage = $request->get('page-no', 1);
        $recordsPerPage = $request->get('records-per-page', config('access.list_page_size'));
        $orderId = $request->get('order', 0);
        $order = new Order();
        $orders = $order->search($request, $currentPage, $recordsPerPage, true);

        if ($request->ajax()) {
            if (!empty($request->get('order'))) {
                $orders = Order::where('user_id', auth()->user()->getCustomerAdmin()->id)
                    ->where('is_saved', 1)
                    ->where('status', '!=', 43)
                    ->where('id', '<=', $request->get('order'))
                    ->orderBy('created_at', 'desc')
                    ->paginate($currentPage);
                $orderId = $request->get('order');
            } else {
                $order = new Order();
                $orders = $order->search($request, $currentPage, $recordsPerPage, true);
            }

            $totalRecords = $orders->total();
            $loadedRecords = $orders->count();
            return response()->json([
                'html' => view('partials.orderlist', [
                    'orders' => $orders,
                    'pageCount' => $currentPage,
                    'RecordsPerPage' => $recordsPerPage,
                    'OrderId' => $orderId,
                ])->render(),
                'totalRecords' => $totalRecords,
                'loadedRecords' => $loadedRecords
            ]);
        }

        $statuses = OrderStatus::getCustomerOrderStatus();
        $statusData = collect($statuses)->map(fn($status) => [
            'id' => $status[0]['id'] ?? null,
            'text' => $status[0]['title'] ?? ''
        ])->filter()->values()->all();

        return app(CommonController::class)->renderOrdersView([
            'layout' => 'frontend.layouts.split',
            'title' => 'Orders' . ' '.html_entity_decode("&#8212;", ENT_QUOTES, 'UTF-8').' ' . app_name(),
            'isBackend' => false,
            'searchFormInclude' => 'frontend.auth.orders.includes.orders-search-form',
            'pageType' => 1,
            'orders' => $orders,
            'pageCount' => $recordsPerPage,
            'orderId' => $orderId,
            'orderData' => null,
            'commentsData' => null,
            'statuses' => $statusData,
        ]);
    }

    /**
     * Show the form for creating a new RX Lens order.
     *
     * @return View
     */
    public function createRXLensOrder(ManageCreateOrderRequest $request): View
    {
        $user = auth()->user()->getCustomerAdmin();

        if ($request->ajax()) {
            return view('frontend.auth.orders.create.rx-form')
                ->withCombinationArray(RxOrders::getCombinationArray())
                ->withUser($user);
        }

        if (!empty($request->get('reuse'))) {
            $orderModal = Order::where('user_id', $user->getCustomerAdmin()->id)
                ->where('id', $request->get('reuse'))
                ->first();
            if ($orderModal) {
                $editData = [
                    'id' => $request->get('reuse'),
                    'reason' => '',
                    'notes' => '',
                    'is_edit' => true,
                    'order' => $orderModal,
                    'type' => 'reuse'
                ];
                return view('frontend.auth.orders.create.rx-lens-order')
                    ->withCombinationArray(RxOrders::getCombinationArray())
                    ->withCombinationHideArray(RxOrders::getCombinationHideArray())
                    ->withUser($user)
                    ->withEditData($editData);
            }
        } elseif (!empty($request->get('edit'))) {
            $orderModal = Order::where('user_id', $user->getCustomerAdmin()->id)
                ->where('id', $request->get('edit'))
                ->first();
            if ($orderModal && $orderModal->isSaved()) {
                $editData = [
                    'id' => $request->get('edit'),
                    'reason' => '',
                    'notes' => '',
                    'is_edit' => true,
                    'order' => $orderModal,
                    'type' => 'edit'
                ];
                return view('frontend.auth.orders.create.rx-lens-order')
                    ->withCombinationArray(RxOrders::getCombinationArray())
                    ->withCombinationHideArray(RxOrders::getCombinationHideArray())
                    ->withUser($user)
                    ->withEditData($editData);
            }
        }

        return view('frontend.auth.orders.create.rx-lens-order')
            ->withCombinationArray(RxOrders::getCombinationArray())
            ->withCombinationHideArray(RxOrders::getCombinationHideArray())
            ->withUser($user);
    }

    /**
     * Show the form for creating a new Stock lens order.
     *
     * @return View
     */
    public function createStocklenseOrder(ManageStockOrderRequest $request): View
    {
        $user = auth()->user()->getCustomerAdmin();
        $material = Stockvalidation::distinct()->get(['material']);
        return view('frontend.auth.orders.create.stock-lens-order')
            ->withMaterial($material)
            ->withUser($user);
    }

    /**
     * Show the form for creating a new Consumable Lens order.
     *
     * @return View
     */
    public function createConsumableLensOrder(ManageCreateOrderRequest $request): View
    {
        $consumableProducts = [];
        $currentParent = null;
        $imported_consumables_orders = ImportedConsumablesOrder::get();

        foreach ($imported_consumables_orders as $product) {
            if ($product['variant'] == 0) {
                $parent = [
                    'id' => $product['id'],
                    'category' => $product['category'],
                    'sub_category' => $product['sub_category'],
                    'upc' => $product['upc'],
                    'name' => $product['name'],
                    'sub_name' => $product['sub_name'],
                    'size_color' => $product['size_color'],
                    'unit' => $product['unit'],
                    'price' => $product['price'],
                    'price_unit' => $product['price_unit'],
                    'description' => $product['description'],
                    'min' => $product['min'],
                    'max' => $product['max'],
                    'image' => self::validateImage($product['image'] ?? 'water-mark.png', 'main'),
                    'image_hover' => self::validateImage($product['image_hover'] ?? '', 'child'),
                    'child' => [],
                ];
                $consumableProducts[] = $parent;
                $currentParent = &$consumableProducts[count($consumableProducts) - 1];
            } else {
                if ($currentParent !== null) {
                    $childData = [
                        'id' => $product['id'],
                        'name' => $product['name'],
                        'sub_name' => $product['sub_name'],
                        'price' => $product['price'],
                        'price_unit' => $product['price_unit'],
                        'description' => $product['description'],
                        'min' => $product['min'],
                        'max' => $product['max'],
                        'image' => self::validateImage($product['image'] ?? 'water-mark.png', 'main'),
                        'image_hover' => self::validateImage($product['image_hover'] ?? '', 'child'),
                    ];

                    if (empty($currentParent['child'])) {
                        array_unshift($currentParent['child'], [
                            'id' => $currentParent['id'],
                            'name' => $currentParent['name'],
                            'sub_name' => $currentParent['sub_name'],
                            'price' => $currentParent['price'],
                            'price_unit' => $currentParent['price_unit'],
                            'description' => $currentParent['description'],
                            'min' => $currentParent['min'],
                            'max' => $currentParent['max'],
                            'image' => self::validateImage($currentParent['image'] ?? 'water-mark.png', 'main'),
                            'image_hover' => self::validateImage($currentParent['image_hover'] ?? '', 'child'),
                        ]);
                    }
                    $currentParent['child'][] = $childData;
                }
            }
        }

        foreach ($consumableProducts as &$parent) {
            if (empty($parent['child'])) {
                $parent['child'][] = [
                    'id' => $parent['id'],
                    'name' => $parent['name'],
                    'sub_name' => $parent['sub_name'],
                    'price' => $parent['price'],
                    'price_unit' => $parent['price_unit'],
                    'description' => $parent['description'],
                    'min' => $parent['min'],
                    'max' => $parent['max'],
                    'image' => self::validateImage($parent['image'] ?? 'water-mark.png', 'main'),
                    'image_hover' => self::validateImage($parent['image_hover'] ?? '', 'child'),
                ];
            }
        }

        $user = auth()->user()->getCustomerAdmin();
        return view('frontend.auth.orders.create.consumable-lens-order')
            ->withConsumableProducts($consumableProducts)
            ->withUser($user);
    }

    private function validateImage(string $imagePath, string $type): string
    {
        if (empty($imagePath)) {
            return '';
        }
        $fullPath = storage_path('app/public/imports/images/' . $imagePath);
        if ($type == 'main') {
            return file_exists($fullPath) ? $imagePath : 'water-mark.png';
        } else {
            return file_exists($fullPath) ? $imagePath : '';
        }
    }

    /**
     * Show the form for creating a new Consumable Lens order.
     *
     * @return View
     */
    public function redo(ManageCreateOrderRequest $request): View
    {
        $user = auth()->user()->getCustomerAdmin();
        $post = $request->all();
        $orderModal = Order::where('user_id', $user->getCustomerAdmin()->id)
            ->where('id', $post['id'])
            ->first();

        if (empty($orderModal)) {
            return redirect()->route('frontend.user.orders')
                ->withFlashDanger(__('Invalid Order'));
        }

        $editData = [
            'id' => $post['id'],
            'reason' => $post['reason'],
            'notes' => $post['notes'],
            'is_edit' => Order::isEditableWithReason($post['reason']),
            'order' => $orderModal,
            'type' => 'redo'
        ];

        return match ($orderModal->type) {
            Order::ORDER_RX => view('frontend.auth.orders.create.rx-lens-order')
                ->withCombinationArray(RxOrders::getCombinationArray())
                ->withCombinationHideArray(RxOrders::getCombinationHideArray())
                ->withUser($user)
                ->withEditData($editData),
            Order::ORDER_STOCK => view('frontend.auth.orders.create.stock-lens-order')
                ->withMaterial(Stockvalidation::distinct()->get(['material']))
                ->withUser($user)
                ->withEditData($editData),
            Order::ORDER_CONSUMABLE => view('frontend.auth.orders.create.consumable-lens-order')
                ->withImportedConsumables(ImportedConsumablesOrder::paginate(10))
                ->withUser($user)
                ->withEditData($editData),
            default => view('frontend.auth.orders.create.rx-lens-order')
                ->withCombinationArray(RxOrders::getCombinationArray())
                ->withCombinationHideArray(RxOrders::getCombinationHideArray())
                ->withUser($user)
                ->withEditData($editData),
        };
    }

    /**
     * Get the combinations for stock order
     *
     * @return mixed
     */
    public function getStockData(Request $request)
    {
        if ($request->ajax()) {
            return match ($request->type) {
                "colour" => Stockvalidation::where('material', $request->get('material'))
                    ->distinct()
                    ->orderBy('id', 'asc')
                    ->get('colour'),
                "coating" => Stockvalidation::where('material', $request->get('material'))
                    ->where('colour', $request->get('colour'))
                    ->distinct()
                    ->orderBy('id', 'asc')
                    ->get('coating'),
                "sph" => Stockvalidation::where('material', $request->get('material'))
                    ->where('colour', $request->get('colour'))
                    ->where('coating', $request->get('coating'))
                    ->distinct()
                    ->orderBy('sph', 'desc')
                    ->get('sph'),
                "cyl" => Stockvalidation::where('material', $request->get('material'))
                    ->where('colour', $request->get('colour'))
                    ->where('coating', $request->get('coating'))
                    ->where('sph', $request->get('sph'))
                    ->distinct()
                    ->orderBy('cyl', 'desc')
                    ->get('cyl'),
                "size" => Stockvalidation::where('material', $request->get('material'))
                    ->where('colour', $request->get('colour'))
                    ->where('coating', $request->get('coating'))
                    ->where('sph', $request->get('sph'))
                    ->where('cyl', $request->get('cyl'))
                    ->get(['size', 'price', 'barcode', 'vendor', 'cards']),
                default => collect(),
            };
        }
    }

    /**
     * Creates New Stock Order
     *
     * @return View
     */
    public function confirmStock(StoreStockOrderRequest $request): View
    {
        $order = $request->all();
        return view('frontend.auth.orders.confirm.confirm-stock')
            ->withOrder($order);
    }

    /**
     * Creates New Stock Order
     *
     * @return RedirectResponse
     */
    public function storeRx(StoreRxOrderRequest $request): RedirectResponse
    {
        $order = Order::newOrder($request->all(), Order::ORDER_RX);

        if ($order) {
            $rxOrder = RxOrders::newRxOrder($order, $request);
            if ($order->isSaved()) {
                return redirect()->route('frontend.user.saved-orders', ['order' => $order->id])
                    ->withFlashSuccess(__('RX Order Saved Successfully'));
            } else {
                $order->uploadPDFToDropbox();
                return redirect()->route('frontend.user.order.confirm', $order->id)
                    ->withFlashSuccess(__('RX Order Created Successfully'));
            }
        }

        return redirect()->back()->withFlashDanger(__('Order creation failed'));
    }

    /**
     * @return View
     */
    public function confirm(Request $request, Order $order): View
    {
        if ($order->user_id == auth()->user()->getCustomerAdmin()->id) {
            return view('frontend.auth.orders.confirm.rx')
                ->withOrder($order);
        }
        return redirect()->route('frontend.user.orders')
            ->withFlashDanger(__('Invalid Request'));
    }

    /**
     * Creates New Stock Order
     *
     * @return RedirectResponse
     */
    public function newStockOrder(StoreStockOrderRequest $request): RedirectResponse
    {
        $order = Order::newOrder($request->all(), 'stock');
        if ($order) {
            $stockOrder = StockOrder::newStockOrder($order, $request->all());
            $order->uploadPDFToDropbox();
        }
        return redirect()->route('frontend.user.order.confirm', $order->id)
            ->withFlashSuccess(__('Stock Order Created Successfully'));
    }

    /**
     * Creates New Consumables Order
     *
     * @return View|JsonResponse
     */
    public function newConsumablesOrder(Request $request): View|JsonResponse
    {
        $confirmOrder = $request->input('confirm_order', false);
        if ($confirmOrder == 'true') {
            $orderData = $request->input('orderData');
            $order = Order::newOrder($orderData, 'consumables', $request->input('ordertotalAmount'));
            if ($order) {
                foreach ($orderData as $key => $value) {
                    $model = ImportedConsumablesOrder::where('id', $value['itemId'])->first();
                    ConsumablesOrder::create([
                        'order_id' => $order->id,
                        'name' => $model->name,
                        'sub_name' => $model->sub_name,
                        'description' => $model->description,
                        'image' => $model->image,
                        'image_hover' => $model->image_hover,
                        'upc' => $model->upc,
                        'quantity' => $value["itemQty"],
                        'price' => $value["basePrice"],
                    ]);
                }
                $order->uploadPDFToDropbox();
                return response()->json(['order_id' => $order->id, 'message' => 'Consumables Order Created Successfully']);
            } else {
                return response()->json(['message' => 'Consumables Order Not Created Successfully']);
            }
        } else {
            $order = ImportedConsumablesOrder::find($request->get('id'));
            return view('frontend.auth.orders.confirm.confirm-consumables')
                ->withOrder($order)
                ->withRequest($request);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return void
     */
    public function store(Request $request): void
    {
        //
    }

    /**
     * Display the specified resource.
     *
     * @return View
     */
    public function show(Request $request, Order $order): View
    {
        return match ($order->type) {
            Order::ORDER_STOCK => view('frontend.auth.orders.show.stock')
                ->withModal($order)
                ->withStockOrder(StockOrder::where('order_id', $order->id)->first()),
            Order::ORDER_CONSUMABLE => view('frontend.auth.orders.show.consumable')
                ->withModal($order),
            Order::ORDER_RX => (function () use ($order) {
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                $rxOrderArray = $rxOrder->getShowData();
                return view('frontend.auth.orders.show.rx')
                    ->withModal($order)
                    ->withRxOrder($rxOrder)
                    ->withRxOrderArray($rxOrderArray);
            })(),
            default => view('frontend.auth.orders.show.rx')
                ->withModal($order)
                ->withRxOrder(null)
                ->withRxOrderArray([]),
        };
    }

    /**
     * Display a listing of the resource.
     *
     * @return View|JsonResponse
     */
    public function all(Request $request): View|JsonResponse
    {
        $pageCount = config('access.list_page_size');
        if (!empty($request->get('page-no'))) {
            $pageCount = $request->get('page-no') + $pageCount;
        }

        $orders = Order::where('user_id', auth()->user()->getCustomerAdmin()->id);
        if (!empty(json_decode($request->get('types')))) {
            $orders = $orders->whereIn('type', json_decode($request->get('types')));
        }

        $orders = $orders->orderBy('created_at', 'desc')->paginate($pageCount);
        if ($request->ajax()) {
            return view('frontend.auth.orders.includes.all')->withOrders($orders)->withPageCount($pageCount);
        }

        return view('frontend.auth.orders.all')
            ->withOrders($orders)->withPageCount($pageCount);
    }

    /**
     * Display a listing of the resource.
     *
     * @return View|JsonResponse
     */
    public function stockLensOrders(ManageStockOrderRequest $request): View|JsonResponse
    {
        $pageCount = config('access.list_page_size');
        if (!empty($request->get('page-no'))) {
            $pageCount = $request->get('page-no') + $pageCount;
        }

        $orders = Order::where('type', Order::ORDER_STOCK)->where('user_id', auth()->user()->getCustomerAdmin()->id)->orderBy('created_at', 'desc')->paginate($pageCount);
        if ($request->ajax()) {
            return view('partials.orderlist')->with('orders', $orders)->with('pageCount', $pageCount);
        }

        return view('frontend.auth.orders.stock-orders')
            ->withOrders($orders)->withPageCount($pageCount)->withOrderType(Order::ORDER_STOCK);
    }

    /**
     * Display a listing of the resource.
     *
     * @return View|JsonResponse
     */
    public function consumableLensOrders(Request $request): View|JsonResponse
    {
        $pageCount = config('access.list_page_size');
        if (!empty($request->get('page-no'))) {
            $pageCount = $request->get('page-no') + $pageCount;
        }

        $orders = Order::where('type', Order::ORDER_CONSUMABLE)->where('user_id', auth()->user()->getCustomerAdmin()->id)->orderBy('created_at', 'desc')->paginate($pageCount);
        if ($request->ajax()) {
            return view('partials.orderlist')->with('orders', $orders)->with('pageCount', $pageCount);
        }

        return view('frontend.auth.orders.consumable-orders')
            ->withOrders($orders)->withPageCount($pageCount)->withOrderType(Order::ORDER_CONSUMABLE);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @return void
     */
    public function edit(Order $order): void
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @return void
     */
    public function update(Request $request, Order $order): void
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @return RedirectResponse
     */
    public function destroy(Order $order): RedirectResponse
    {
        if ($order) {
            $order->delete();
            return redirect()->route('frontend.user.saved-orders')
                ->withFlashSuccess(__('Saved Order Deleted'));
        } else {
            return redirect()->route('frontend.user.orders')
                ->withFlashDanger(__('Invalid Order'));
        }
    }

    public function searchOrders(Order $order, Request $request): JsonResponse
    {
        $currentPage = $request->get('page-no', 1);
        $recordsPerPage = $request->get('records-per-page', config('access.list_page_size'));
        if ($request->ajax()) {
            $orders = $order->search($request, $currentPage, $recordsPerPage);
            $totalRecords = $orders->total();
            $loadedRecords = $orders->count();
            return response()->json([
                'html' => view('partials.orderlist', [
                    'isBackend' => false,
                    'orders' => $orders,
                    'pageCount' => $currentPage,
                    'RecordsPerPage' => $recordsPerPage,
                    'orderType' => $request->get('order_type')
                ])->render(),
                'totalRecords' => $totalRecords,
                'loadedRecords' => $loadedRecords
            ]);
        }
    }

    /**
     * Search Barcode Api.
     *
     * @return JsonResponse
     */
    public function searchBarcodeApi(Request $request): JsonResponse
    {
        if ($request->ajax() && !empty($request->get('value'))) {
            $data = Stockvalidation::where('barcode', str_replace(' ', '', $request->get('value')))->first();
            if ($data) {
                return response()->json(['status' => true, 'data' => $data]);
            }
        }
        return response()->json(['status' => false, 'message' => ['Error! Invalid Barcode Number']]);
    }

    public function newBarcodeOrder(Request $request): JsonResponse
    {
        $rules = [
            'id' => ['required', 'exists:stock_validation,id']
        ];
        $validator = Validator::make($request->all(), $rules);
        $errors = [];
        if ($validator->passes()) {
            $model = Stockvalidation::where('id', $request->get('id'))->first();
            $order = Order::newOrder($model, Order::ORDER_BARCODE);
            if ($order) {
                $stockOrder = StockOrder::newBarCodeOrder($order, $model);
                $order->uploadPDFToDropbox();
                return response()->json(['success' => 'Successfully saved.']);
            }
        } else {
            $errors = $validator->errors()->all();
        }

        return response()->json(['error' => $errors]);
    }

    public function rxValidationApi(Request $request): JsonResponse
    {
        if ($request->ajax()) {
            return response()->json(['status' => true, 'data' => RxValidations::validateData($request->all())]);
        }
        return response()->json(['status' => false, 'message' => ['Error! Something went wrong.']]);
    }

    public function rxFilteredValidationApi(Request $request): JsonResponse
    {
        if ($request->ajax()) {
            return response()->json(['status' => true, 'data' => RxValidations::getFilteredValidationData($request->all())]);
        }
        return response()->json(['status' => false, 'message' => ['Error! Something went wrong.']]);
    }

    public function filterFrameInfo(Request $request): JsonResponse
    {
        if ($request->ajax()) {
            $data = RxValidations::filterFrameInfo($request->get('type'), $request->get('brand'), $request->get('model'), $request->get('colour'), $request->get('size'), $request->get('package'), $request->get('is_only_price'), $request->get('statusD'));
            return response()->json(['status' => true, 'data' => $data['list'], 'metaData' => $data['meta_data']]);
        }
        return response()->json(['status' => false, 'message' => ['Error! Something went wrong.']]);
    }

    public function rxValidationCutoutApi(Request $request): JsonResponse
    {
        if ($request->ajax()) {
            $status = false;
            $message = '';
            $size = '';
            $stock = Stockvalidation::where('material', RxValidations::decodeValue($request->get('lens_material')))
                ->where('colour', RxValidations::decodeValue($request->get('lens_colour')))
                ->where('coating', RxValidations::decodeValue($request->get('coating_name')))
                ->where('sph', $request->get('sph'))
                ->where('cyl', $request->get('cyl'))
                ->first();
            if ($stock) {
                $status = true;
                $size = $stock->size;
            }
            return response()->json(['status' => $status, 'message' => '', 'size' => $size]);
        }
        return response()->json(['status' => false, 'message' => ['Error! Something went wrong.']]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @return JsonResponse
     */
    public function copy(Request $request): JsonResponse
    {
        if ($request->ajax()) {
            $model = Order::where('id', $request->get('id'))->where('user_id', auth()->user()->getCustomerAdmin()->id)->first();
            if ($model) {
                $newModel = $model->makeCopy();
                if ($newModel) {
                    $newModel->uploadPDFToDropbox();
                    return response()->json(['status' => true, 'message' => 'Order Created Successfully', 'id' => $newModel->id]);
                }
            }
        }
        return response()->json(['status' => false, 'message' => ['Error! Something went wrong. Please try again']]);
    }

    /**
     * Display the specified resource.
     *
     * @return View
     */
    public function print(Request $request, Order $order): View
    {
        $user = auth()->user();
        if ($order->user_id != $user->getCustomerAdmin()->id) {
            // Invalid order, but continue for now
        }

        return match ($order->type) {
            Order::ORDER_RX => (function () use ($order) {
                $user = $order->user;
                $profile = $order->user->profile;
                $rxOrder = RxOrders::where('order_id', $order->id)->first();
                $rxOrderArray = $rxOrder->getShowData();
                return view('frontend.auth.orders.print.rx')
                    ->withOrder($order)
                    ->withRxOrder($rxOrder)
                    ->withUser($user)
                    ->withProfile($profile)
                    ->withRxOrderArray($rxOrderArray);
            })(),
            Order::ORDER_BARCODE, Order::ORDER_STOCK => (function () use ($order) {
                $stockOrder = StockOrder::where('order_id', $order->id)->first();
                $user = $order->user;
                $profile = $order->user->profile;
                return view('frontend.auth.orders.print.stock')
                    ->withModal($order)
                    ->withUser($user)
                    ->withProfile($profile)
                    ->withStockOrder($stockOrder);
            })(),
            Order::ORDER_CONSUMABLE => (function () use ($order) {
                $user = $order->user;
                $profile = $order->user->profile;
                return view('frontend.auth.orders.print.consumable')
                    ->withOrder($order)
                    ->withUser($user)
                    ->withProfile($profile);
            })(),
            default => view('frontend.auth.orders.print.rx')
                ->withOrder($order)
                ->withRxOrder(null)
                ->withUser($order->user)
                ->withProfile($order->user->profile)
                ->withRxOrderArray([]),
        };
    }

    public function orderFeedback(Request $request): JsonResponse
    {
        $feedback = $request->input('feedback');
        $orderId = $request->input('orderId');
        try {
            $order = Order::findOrFail($orderId);
            $order->update([
                'is_feedback' => 1
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Order not found']);
        }
        $htmlTable = self::generateHtmlTable($feedback);
        $currentDate = date('d-M-Y h:i a');
        $user = auth()->user();
        $accountCode = $user->ac_code ?: '';
        $replyToEmail = $user->email ?: '';
        $businessName = $user->profile->business_name ?: '';
        $subject = 'Order Platform Feedback, ' . htmlspecialchars($accountCode) . ' - ' . htmlspecialchars($businessName) . ' ' . $currentDate;
        $emailTo = 'info@lens-shapers.com';
        sendEmail($htmlTable, $emailTo, $replyToEmail, $subject);
        return response()->json([
            'success' => true,
            'is_feedback' => true,
            'message' => 'Thank you for your feedback',
        ]);
    }

    private function generateHtmlTable(string $feedback): string
    {
        $html = '<table style="border-collapse: collapse; width: 50%;">';
        $html .= '<tr style="background-color: #f2f2f2;">';
        $html .= '<th style="border: 1px solid #ddd; padding: 8px;">Feedback</th>';
        $html .= '</tr>';
        $html .= '<tr>';
        $html .= '<td style="border: 1px solid #ddd; padding: 8px;">' . htmlspecialchars($feedback) . '</td>';
        $html .= '</tr>';
        $html .= '</table>';
        return $html;
    }

    public function detailsTypeApi(Request $request)
    {
        return $this->commentService->handleRequest($request, false);
    }

    private function getOrderDetails($orderId)
    {
        $order = Order::with(['user'])->find($orderId);

        if (!$order) return null;

        $type = $order->type;
        $isCustomerAdmin = auth()->user()->isCustomerAdmin();
        $stockOrder = null;
        $rxOrder = null;
        $rxOrderArray = [];
        $consumableOrder = null;

        if ($order->type == Order::ORDER_STOCK) {
            $stockOrder = \App\Models\Backend\StockOrder::where('order_id', $orderId)->first();
            return view('partials.stock', compact('order', 'stockOrder', 'isCustomerAdmin'))->render();
        } elseif ($order->type == Order::ORDER_RX) {
            $rxOrder = \App\Models\Backend\RxOrders::where('order_id', $orderId)->first();
            if ($rxOrder) {
                $rxOrderArray = $rxOrder->getShowData();
            }
            return view('partials.rx', compact('order', 'rxOrder', 'rxOrderArray', 'isCustomerAdmin'))->render();
        } elseif ($order->type == Order::ORDER_CONSUMABLE) {
            $consumableOrder = \App\Models\Backend\ConsumableOrder::where('order_id', $orderId)->first();
            return view('partials.consumable', compact('order', 'consumableOrder', 'isCustomerAdmin'))->render();
        }

        return '<div>No details available for this order type.</div>';
    }

    private function getOrderComments($orderId)
    {
        $order = Order::find($orderId);

        if (!$order) return null;

        $logs = PhoneLog::with(['replies' => function($query) {
            $query->with('user')->orderBy('created_at', 'asc');
        }])
        ->where('order_id', $orderId)
        ->whereNull('replied')
        ->when(!auth()->user()->isCustomerAdmin(), function($query) {
            $query->where('type', '!=', 'notes');
        })
        ->orderBy('created_at', 'desc')
        ->get();

        $user = auth()->user();
        $isCustomerAdmin = $user->isCustomerAdmin();
        $type = $order->type;
        $stockOrder = null;
        $rxOrder = null;
        $rxOrderArray = null;

        if ($order->type == Order::ORDER_STOCK) {
            $stockOrder = \App\Models\Backend\StockOrder::where('order_id', $orderId)->first();
        }

        if ($order->type == Order::ORDER_RX) {
            $rxOrder = \App\Models\Backend\RxOrders::where('order_id', $orderId)->first();
            $rxOrderArray = $rxOrder ? $rxOrder->getShowData() : null;
        }

        return view('partials.chatbot_grid', [
            'logs' => $logs,
            'user' => $user,
            'sender' => 'admin',
            'showButtons' => false,
            'isCustomerAdmin' => $isCustomerAdmin,
            'type' => $type,
            'order' => $order,
            'stockOrder' => $stockOrder,
            'rxOrder' => $rxOrder,
            'rxOrderArray' => $rxOrderArray,
        ])->render();
    }
}