<?php

namespace App\Http\Controllers\Frontend\Auth;

use App\Models\Auth\User;
use Illuminate\Support\Str;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Password;
use Illuminate\Auth\Events\PasswordReset;
use App\Repositories\Frontend\Auth\UserRepository;
use App\Http\Requests\Frontend\User\ManageUserRequest;
use App\Http\Requests\Frontend\Auth\ResetPasswordRequest;
use App\Http\Requests\Frontend\User\UpdateUserPasswordRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;

/**
 * Class ResetPasswordController.
 */
class ResetPasswordController extends Controller
{
    /**
     * @var UserRepository
     */
    protected UserRepository $userRepository;

    /**
     * ChangePasswordController constructor.
     *
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * Display the password reset view for the given token.
     *
     * If no token is present, display the link request form.
     *
     * @param string|null $token
     *
     * @return View|RedirectResponse
     */
    public function showResetForm(?string $token = null): View|RedirectResponse
    {
        if (!$token) {
            return redirect()->route('frontend.auth.password.email');
        }

        $ac_code = request('ac_code');
        $business_name = null;
        $email = null;

        if ($ac_code) {
            // Find password reset record with ac_code
            $passwordReset = DB::table('password_resets')
                ->where('token', $token)
                ->where('ac_code', $ac_code)
                ->first();

            if (!$passwordReset) {
                return redirect()->route('frontend.auth.password.email')->withErrors(['ac_code' => 'Invalid reset link or Customer Code.']);
            }

            $email = $passwordReset->email;
            $user = User::where('email', $passwordReset->email)
                ->where('ac_code', $ac_code)
                ->first();

            if ($user) {
                $business_name = $user->business_name;
            }
        } else {
            // Legacy handling without ac_code
            $user = $this->userRepository->findByPasswordResetToken($token);

            // If user not found, try to get email from password_resets table
            if (!$user) {
                $passwordReset = DB::table('password_resets')
                    ->where('token', $token)
                    ->whereNull('ac_code')
                    ->first();
                if ($passwordReset) {
                    $email = $passwordReset->email;
                }
            } else {
                $email = $user->email;
            }
        }

        $viewData = [
            'token' => $token,
            'ac_code' => $ac_code,
            'business_name' => $business_name,
            'email' => $email,
        ];

        if (!empty($user) && ($user->isAdmin() || $user->isLabManager())) {
            if ($user && resolve('auth.password.broker')->tokenExists($user, $token)) {
                return view('frontend.auth.passwords.reset')->with($viewData);
            }
        } elseif (!empty($user) && $user->isCustomerAdmin()) {
            if ($user && resolve('auth.password.broker')->tokenExists($user, $token)) {
                return view('frontend.auth.passwords.customer.reset')->with($viewData);
            }
        }

        // For regular customers, show the reset form
        return view('frontend.auth.passwords.reset')->with($viewData);
    }

    /**
     * Get the password reset validation rules.
     *
     * @return array
     */
    protected function rules(): array
    {
        $rules = [
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|confirmed|min:6',
            'ac_code' => request('ac_code') ? 'required|string' : 'nullable|string',
        ];

        return $rules;
    }

    /**
     * Reset the given user's password.
     *
     * @param ResetPasswordRequest $request
     * @return RedirectResponse
     */
    public function reset(ResetPasswordRequest $request): RedirectResponse
    {
        // First, try to find token with ac_code if provided
        $tokenQuery = DB::table('password_resets')->where('email', $request->email);

        if ($request->ac_code) {
            $tokenQuery->where('ac_code', $request->ac_code);
        } else {
            // If no ac_code provided, look for tokens with NULL ac_code (for backward compatibility)
            $tokenQuery->whereNull('ac_code');
        }

        $tokenData = $tokenQuery->first();

        // Check if the token is valid
        if (!$tokenData || !Hash::check($request->token, $tokenData->token)) {
            return back()->withErrors(['ac_code' => 'Invalid token or Customer Code.']);
        }

        // Find the user by email and ac_code if provided
        $userQuery = User::where('email', $request->email);

        if ($request->ac_code) {
            $userQuery->where('ac_code', $request->ac_code);
        }

        $user = $userQuery->first();

        if (!$user) {
            return back()->withErrors(['ac_code' => 'No user found with this Customer Code.']);
        }

        $user->password = $request->password;
        $user->password_changed_at = now();
        $user->setRememberToken(Str::random(60));
        $user->save();

        DB::table('password_resets')->where('email', $request->email)->delete();
        event(new PasswordReset($user));

        $this->guard()->login($user);
        return $this->sendResetResponse(Password::PASSWORD_RESET);
    }

    /**
     * Reset the given user's password.
     *
     * @param \Illuminate\Contracts\Auth\CanResetPassword $user
     * @param string $password
     */
    protected function resetPassword($user, string $password): void
    {
        $user->password = $password;
        $user->password_changed_at = now();
        $user->setRememberToken(Str::random(60));
        $user->save();

        event(new PasswordReset($user));

        $this->guard()->login($user);
    }

    /**
     * Get the response for a successful password reset.
     *
     * @param string $response
     * @return RedirectResponse
     */
    protected function sendResetResponse(string $response): RedirectResponse
    {
        return redirect()->route(home_route())->withFlashSuccess(e(trans($response)));
    }

    /**
     * @param ManageUserRequest $request
     * @param User $user
     *
     * @return View
     */
    public function edit(ManageUserRequest $request, User $user): View
    {
        return view('frontend.user.change-password')
            ->withUser($user);
    }

    /**
     * @param UpdateUserPasswordRequest $request
     * @param User $user
     *
     * @throws \App\Exceptions\GeneralException
     * @return RedirectResponse
     */
    public function update(UpdateUserPasswordRequest $request, User $user): RedirectResponse
    {
        $this->userRepository->updatePassword($request->only('password'), $user);

        return redirect()->route('frontend.user.index')->withFlashSuccess(__('alerts.backend.users.updated_password'));
    }

    /**
     * Get the guard to be used during password reset.
     *
     * @return \Illuminate\Contracts\Auth\StatefulGuard
     */
    protected function guard()
    {
        return \Illuminate\Support\Facades\Auth::guard();
    }
}