<?php

namespace App\Listeners;

use App\Events\OrderPlaced;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use PDF; // Assuming barryvdh/laravel-dompdf
use App\Models\Backend\Order;
use App\Models\Backend\RxOrders;
use App\Models\Backend\StockOrder;
use Illuminate\Support\Facades\Log;
use Exception;
use App\Custom\DropboxComponent;
use App\Models\Backend\OrderStatus;

class UploadOrderPdfListener implements ShouldQueue
{
    use InteractsWithQueue, SerializesModels;

    public $tries = 3; // Retry up to 3 times
    public $timeout = 120; // Timeout after 120 seconds

    protected $labOrderController;
    protected $labOrderApiService;

    public function __construct()
    {
        $this->labOrderController = app()->make(\App\Http\Controllers\API\LabOrderController::class);
        $this->labOrderApiService = app()->make(\App\Services\LabOrderApiService::class);
    }

    public function handle(OrderPlaced $event)
    {
        $order = $event->order;
        if (empty($order)) {
            Log::channel('order_processing')->error("Order not found for ID: {$event->order->id}");
            $this->fail(new Exception("Order not found"));
            return;
        }

        try {
            $html = '';
            $data = [];
            switch ($order->type) {
                case Order::ORDER_RX:
                    $user = $order->user;
                    $profile = $order->user->profile;
                    $rxOrder = RxOrders::where('order_id', $order->id)->first();
                    if (!$rxOrder) {
                        throw new Exception("RxOrder not found for order ID: {$order->id}");
                    }
                    $rxOrderArray = $rxOrder->getShowData();
                    $html = 'backend.auth.orders.includes.print-pdf.rx';
                    $data = [
                        'order' => $order,
                        'rxOrder' => $rxOrder,
                        'user' => $user,
                        'profile' => $profile,
                        'rxOrderArray' => $rxOrderArray,
                    ];
                    break;
                case Order::ORDER_BARCODE:
                case Order::ORDER_STOCK:
                    $stockOrder = StockOrder::where('order_id', $order->id)->first();
                    if (!$stockOrder) {
                        throw new Exception("StockOrder not found for order ID: {$order->id}");
                    }
                    $user = $order->user;
                    $profile = $order->user->profile;
                    $html = 'backend.auth.orders.includes.print-pdf.stock';
                    $data = [
                        'modal' => $order,
                        'user' => $user,
                        'profile' => $profile,
                        'stockOrder' => $stockOrder,
                    ];
                    break;
                case Order::ORDER_CONSUMABLE:
                    $user = $order->user;
                    $profile = $order->user->profile;
                    $html = 'backend.auth.orders.includes.print-pdf.consumable';
                    $data = [
                        'order' => $order,
                        'user' => $user,
                        'profile' => $profile
                    ];
                    break;
                default:
                    Log::channel('order_processing')->error("Invalid order type: {$order->type}");
            }

            if (empty($html)) {
                Log::channel('order_processing')->error("No HTML template defined for order ID: {$order->id}");
            }

            // Process Labzilla integration for non-RX orders
            if ($order->type !== Order::ORDER_RX) {
                $result = null;
                try {
                    switch ($order->type) {
                        case Order::ORDER_STOCK:
                            $result = $this->labOrderController->stockOrderArray($order);
                            break;
                        case Order::ORDER_CONSUMABLE:
                            $result = $this->labOrderController->consumableOrderArray($order);
                            break;
                    }

                    if ($result) {
                        $rxContent = null;
                        if ($order->type === Order::ORDER_CONSUMABLE) {
                            $rxContent = $this->labOrderController->generateConsumableContent($order, $result);
                        } elseif ($order->type === Order::ORDER_STOCK) {
                            $rxContent = $this->labOrderController->generateStockContent($order, $result);
                        }
                        $trContent = $this->labOrderController->generateTrContent();

                        $hashRouting = $this->labOrderApiService->getHashRouting();
                        $hashRoutingValue = $hashRouting['message']['lab']['contractSending'][0]['hash_routing'] ?? null;

                        $orderData = [
                            'order' => [
                                'hash_routing' => $hashRoutingValue,
                                'rx_content' => $rxContent,
                                'tr_content' => $trContent,
                            ]
                        ];
                        $manualClick = false;
                        $labzillaResponse = $this->labOrderApiService->pushOrderToLab($orderData, $order, $manualClick);
                        if (isset($labzillaResponse['message']['id'])) {
                            $id = (string)$labzillaResponse['message']['id'];
                            $data['labzilla_response'] = substr($id, 1);
                        }
                    }
                } catch (Exception $e) {
                    Log::channel('order_processing')->error("Labzilla integration failed for order ID: {$order->id}. Error: {$e->getMessage()}");
                }
            }

            // Generate and save PDF
            try {
                $pdf = PDF::loadView($html, $data);
                $fileName = 'order-' . $order->id . '.pdf';
                $filePath = storage_path('app/public/order-files/' . $fileName);
                $pdf->save($filePath);
            } catch (Exception $e) {
                Log::channel('order_processing')->error("PDF generation failed for order ID: {$order->id}. Error: {$e->getMessage()}");
            }

            // Upload to Dropbox
            try {
                DropboxComponent::uploadPdf($filePath, '/' . $fileName, config('access.api.dropbox.order_pdf_upload_path'));
            } catch (Exception $e) {
                Log::channel('order_processing')->error("Dropbox upload failed for order ID: {$order->id}. Error: {$e->getMessage()}");
            }

            // Update order status
            $order->is_uploaded = 1;
            $order->save();

            $statusModal = OrderStatus::where('name', 'received_at_lab')->first();
            if (!empty($statusModal)) {
                $order->status = $statusModal->id;
                $order->received_at_lab_date = date('Y-m-d H:i:s');
                $order->update();
            }
            Log::channel('order_processing')->info("Order ID: {$order->id} processed successfully.");

        } catch (Exception $e) {
            Log::channel('order_processing')->error("Error processing order ID: {$order->id}. Error: {$e->getMessage()}");
            $this->fail($e); // Trigger retry or permanent failure
        }
    }

    public function failed(OrderPlaced $event, $exception)
    {
        $order = $event->order;
        Log::channel('order_processing')->critical("Order ID: {$order->id} permanently failed. Error: {$exception->getMessage()}");
        // Update order to reflect failure
        $order->update(['is_uploaded' => 0]);
    }
}