<?php

namespace App\Models\Backend;


use Illuminate\Http\Request;
use App\Models\Auth\User;
use App\Models\Backend\Order;
use App\Models\Backend\EmailTemplate;
use Illuminate\Database\Eloquent\Model;
use App\Models\Backend\PhoneLog;

class Notification extends Model
{
    protected $table = "notifications";

    protected $fillable = [
        'id',
        'user_id',
        'order_id',
        'plog_id',
        'title',
        'from',
        'to',
        'has_read',
        'created_at',
        'updated_at',
    ];

    /**
     * @return mixed
     */
    public function fromUser()
    {
        return $this->hasOne(User::class,'id','from');
    }

    /**
     * @return mixed
     */
    public function toUser()
    {
        return $this->hasOne(User::class,'id','to');
    }

    /**
     * @return mixed
     */
    public function order()
    {
        return $this->hasOne(Order::class,'id','order_id');
    }

    /**
     * Store a newly created resource
     *
     * @param  App\Models\Backend\EmailTemplate  $email
     * @param  App\Models\Backend\Order          $order
     * 
     * @return object
     */
    public static function addNew($title,$from,$to,$orderId = null,$logId = null,  $reply = false){
        return  self::create([
            'title'     =>  $title,
            'user_id'    =>  $from,
            'from'      =>  $from,
            'to'        =>  $to,
            'order_id'  =>  $orderId,
            'plog_id'   =>  $logId,
            // 'has_read'  =>  $reply == true ? 1 : 0
        ]);
    }

    public static function updateHasRead($repliedId) 
    {
        return Notification::where('plog_id', $repliedId)
                    ->where('has_read', 0);
                    // ->update(['has_read' => 1]);
    }


    /**
     * Store a newly created resource
     *
     * @param  App\Models\Backend\EmailTemplate  $email
     * @param  App\Models\Backend\Order          $order
     * 
     * @return object
     */
    public function sendToCustomer(Order $order, EmailTemplate $email) : Notification {

        return  parent::create([
            'title'     =>  $email->title,
            'from'      =>  auth()->user()->id,
            'to'        =>  $order->user->id,
        ]);
    }


    /**
     * Store a newly created resource
     *
     * @param  App\Models\Backend\EmailTemplate  $email
     * @param  App\Models\Backend\Order          $order
     * 
     * @return object
     */
    public function sendToAdmin(Order $order, EmailTemplate $email, Request $request) : Notification {

        return  parent::create([
            'title'     =>  $email->title,
            'from'      =>  $order->user_id,
            'to'        =>  $request->to,
        ]);
    }

    /**
     * Get unread notifications by the user
     * 
     * @return Illuminate\Database\Eloquent\Collection
     */ 
    public function getUnread() {
        return parent::where('to',auth()->user()->id)
                 ->where('has_read', 0)
                 ->get();
     }

      /**
     * Read all notification by authenticated user
     *
     * @return bool
     */
    public function readAll() : bool {
        parent::where('to',auth()->user()->id)
            ->where('has_read',0)
            ->update(['has_read' => 1]);

        return true;
    }

    /**
     * @return bool
     */
    public static function readAllByUser($id, $oid, $logId, $type, $isAdmin) {


        $counter = 0;
        if($type == 'comments') {
            return parent::where('to', $id)
            ->where('order_id', $oid)
            ->where('has_read', 0)
            ->update([
                'has_read' => 1
            ]);
        } else {

            if($isAdmin) {

                $phoneLogIds = PhoneLog::where('replied', $logId)->pluck('id');
                $updatedRepliedNotifications = Notification::whereIn('plog_id', $phoneLogIds)
                                                            ->where('has_read', 0)
                                                            ->update(['has_read' => 1]);
                $updatedMainNotification = parent::where('to', $id)
                                                ->where('order_id', $oid)
                                                ->where('plog_id', $logId)
                                                ->where('has_read', 0)
                                                ->update(['has_read' => 1]);
            } else {

                $phoneLogIds = PhoneLog::where('replied', $logId)->pluck('id');
                $updatedRepliedNotifications = Notification::whereIn('plog_id', $phoneLogIds)
                                                            ->where('has_read', 0)
                                                            ->update(['has_read' => 0]);
                $updatedMainNotification = parent::where('to', $id)
                                                ->where('order_id', $oid)
                                                ->where('plog_id', $logId)
                                                ->where('has_read', 0)
                                                ->update(['has_read' => 0]);
            }
            $counter =  $updatedRepliedNotifications + $updatedMainNotification;
            return $counter;
        }
    }

    /**
     * @return bool
     */
    public function sendEmail(){
        $type = 'customer_order_notifications';
        if(!$this->toUser->isCustomerAdmin()){
            $type = 'admin_order_notifications';
        }
        $template = EmailTemplate::where('name',$type)->first();
        if(!empty($template)){
            $order = $this->order;
            $user = $this->fromUser;
            $subject = $template->getSubject($order,$user);
            $body = $template->getDescription($order,$user);
            $actualMessage = $this->title;
            if(!empty($this->plog_id)){
                $phonelog = PhoneLog::where('id',$this->plog_id)->first();
                if(!empty($phonelog)){
                    $actualMessage = $phonelog->description;
                }
            }
            
            $body = $template->parseActualMessage($body,$actualMessage);
            $data = [
                'subject'   =>  $subject,
                'body'   =>  $body
            ];
            if(!$this->fromUser->isCustomerAdmin()){
                \Mail::to($this->toUser->email)
                     // ->replyTo('service@lens-shapers.com', 'Lens Shapers')
                    ->send(new \App\Mail\OrderNotificationMail($data));
            }else{
                \Mail::to($this->toUser->email)
                ->send(new \App\Mail\OrderNotificationMail($data));
            }
            // \Mail::to('akif.izhar@gmail.com')->send(new \App\Mail\OrderNotificationMail($data));
            // \Mail::to('kashif.izhar@gmail.com')->send(new \App\Mail\OrderNotificationMail($data));
            // \Mail::to('usama485@gmail.com')->send(new \App\Mail\OrderNotificationMail($data));
        }
    }

    public function userProfile()
    {
        return $this->belongsTo(UserProfile::class, 'user_id', 'user_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function orders()
    {
        return $this->belongsTo(Order::class, 'user_id', 'user_id');
    }

    public function PhoneLog()
    {
        return $this->belongsTo(PhoneLog::class, 'plog_id', 'id');
    }

}