<?php

namespace App\Models\Backend;

use DB;
use App\Models\Auth\User;
use Illuminate\Http\Request;
use App\Models\Backend\StockOrder;
use App\Models\Backend\RxOrders;
use App\Models\Backend\OrderStatus;
use Illuminate\Database\Eloquent\Model;
use App\Models\Auth\Traits\Attribute\OrderAttribute;

class Order extends Model
{
    use OrderAttribute;

    const ORDER_RX = 'rx';
    const ORDER_STOCK = 'stock';
    const ORDER_CONSUMABLE = 'consumables';
    const ORDER_BARCODE = 'barcode';

    protected $table = "orders";
    protected $casts = [
        'eta' => 'date',
    ];

    protected $fillable = [
        'id',
        'user_id',
        'patient',
        'is_multipair',
        'multipair_ref',
        'type',
        'method',
        'status',
        'price',
        'tray',
        'invoice_no',
        'is_uploaded',
        'shipping_carrier',
        'tracking_no',
        'shipped_date',
        'order_csv',
        'online',
        'eta',
        'reason_oid',
        'reason_type',
        'reason_notes',
        'material_code',
        'lens_code',
        'colour_code',
        'vendor_code',
        'coating_code',
        'multiple',
        'status_date_time',
        'is_feedback',
        'is_saved',
        'saved_at',
        'received_at_lab_date',
        'created_at',
        'updated_at'
    ];

    /**
     * @return mixed
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * @return mixed
     */
    public function consumableModal()
    {
        return $this->hasOne(ConsumablesOrder::class,'order_id','id');
    }

    /**
     * @return mixed
     */
    public function consumableModals()
    {
        return $this->hasMany(ConsumablesOrder::class,'order_id','id');
    }

    public function getOrderNumber(){
        if($this->isSaved()){
            return 'S'.$this->id;
        }
        if(!empty($this->order_csv))
            return $this->order_csv;

        return $this->method.$this->id;
    }

    public function isMultiple(){
        return $this->multiple;
    }

    public function getOfflineOrderNumber(){
        if($this->isSaved()){
            return 'S'.$this->id;
        }
        if(!empty($this->order_csv))
            return $this->order_csv;

        return $this->method.$this->id;
    }

    public function getMethodIcon(){
        switch (strtolower($this->method)) {
            case 'f':
                return '<i class="fa fa-fax" aria-hidden="true"></i>';
                break;
            case 'd':
                return '<i class="fa fa-wifi" aria-hidden="true"></i>';
                break;
            default:
                // return '<i class="fa fa-globe" aria-hidden="true"></i>';
                break;
            }
    }

    // public function getMethodIcon()
    // {
    //     $orderType = $this->type ?? null;
    //     $method = strtolower($this->method);
    //     $isOnline = $this->online ?? false;

    //     $getRxIcon = function ($lensType) {
    //         $icons = [
    //             'Single_Vision_Finished' => 'rx-single-vision',
    //             'Single_Vision_Surfaced_Conventional' => 'rx-surfaced-conventional',
    //             'Single_Vision_Surfaced_Digital_HD' => 'rx-surfaced-digital',
    //             'Single_Vision_AntiminusFatigue' => 'rx-anti-fatigue',
    //             'Lined_Multiminusfocal' => 'rx-lined-multifocal',
    //             'Progressive' => 'rx-progressive',
    //             'Computer_Progressive' => 'rx-computer-progressive',
    //             'Myopia_Management' => 'rx-myopia',
    //             'Extended_Computer_Progressive' => 'rx-extended-computer-progressive',
    //             'Special' => 'rx-special',
    //         ];
    //         $icon = $icons[$lensType] ?? 'rx-single-vision';
    //         return '<img class="svg-icons-order" src="' . asset("public/storage/svg/{$icon}.svg") . '" alt="RX ' . ($lensType ?: 'Single Vision') . '">';
    //     };

    //     if ($isOnline) {
    //         if ($orderType === self::ORDER_RX) {
    //             if(isset($this->rxOrder->data)) {
    //                 return $getRxIcon(json_decode($this->rxOrder->data, true)['lens']['lens_type'] ?? '');
    //             }
    //         }
    //         if ($orderType === self::ORDER_STOCK) {
    //             return svg_image('order-type-stock.svg', 'svg-icons-order', 'Stock');
                        
    //         }
    //         if ($orderType === self::ORDER_CONSUMABLE) {
    //             return svg_image('consumable.svg', 'svg-icons-order', 'Consumable');
    //         }
    //     } else {
    //         switch ($method) {
    //             case 'f':
    //             case 'p':
    //                 return svg_image('order-type-fax.svg', 'svg-icons-order', 'Fax');
    //             case 'd':
    //             case 'c':
    //                 return svg_image('order-type-courier.svg', 'svg-icons-order', 'Courier');
    //             default:
    //                 if ($orderType === self::ORDER_STOCK) {
    //                     return svg_image('order-type-stock.svg', 'svg-icons-order', 'Stock');
    //                 }
    //                 if ($orderType === self::ORDER_CONSUMABLE) {
    //                     return svg_image('order-type-consumable.svg', 'svg-icons-order', 'Consumable');
    //                 }
    //                 if ($orderType === self::ORDER_RX) {
    //                     if(isset($this->rxOrder->data)) {
    //                         return $getRxIcon(json_decode($this->rxOrder->data, true)['lens']['lens_type'] ?? '');
    //                     }
    //                 }
    //         }
    //     }
    //     return '';
    // }


    public function getTypeIcon(){
        $src = url('public/img/rx-icon.png');
        $type = $this->type;
        if($this->isFax() || $this->isPhone())
            $type = 'f/p';
        elseif($this->isCourier())
            $type = 'courier';

        switch ($type) {
            case 'f/p':
                $src = url('public/img/orders/fax-phone_order.svg');
                break;
            case 'courier':
                $src = url('public/img/orders/courier_order.svg');
                break;
            case self::ORDER_STOCK:
                $src = url('public/img/orders/sv-g_(stock).svg');
                break;
            case self::ORDER_CONSUMABLE:
                $src = url('public/img/orders/consumeable.svg');
                break;
            case self::ORDER_RX:
                if(!empty($this->rxOrder->data)){
                    $metaData = json_decode($this->rxOrder->data,true);
                    if(isset($metaData['lens'])){
                        $lensType = strtolower(RxValidations::decodeValue($metaData['lens']['lens_type']));
                        $subType = isset($metaData['lens']['sub_type']) ?? strtolower(RxValidations::decodeValue($metaData['lens']['sub_type']));
                        if(str_contains($lensType,'omega rs'))
                            $src = url('public/img/orders/rs.svg');
                        if(str_contains($lensType,'progressive'))
                            $src = url('public/img/orders/progressive.svg');
                        if(str_contains($lensType,'single vision') && str_contains($subType,'surfaced'))
                            $src = url('public/img/orders/sv-sf.svg');
                        elseif(str_contains($lensType,'multi-focal') && str_contains($subType,'flat top'))
                            $src = url('public/img/orders/flat-top.svg');
                        elseif(str_contains($lensType,'multi-focal') && str_contains($subType,'executive'))
                            $src = url('public/img/orders/executive.svg');
                    }
                }
                break;
        }
        return '<img src="'.$src.'" class="order-type-icon">';
    }


    /**
     * @return string
     */
    public function getType()
    {
        return ucwords($this->type);
    }

    public function getOrderDate($format = 'd-M-Y h:i a'){
        return date_time_format($this->created_at,$format);
    }

    public function isFax(){
        return strtolower($this->method) == 'f' ? true : false;
    }

    public function isPhone(){
        return strtolower($this->method) == 'p' ? true : false;
    }

    public function isCourier(){
        return strtolower($this->method) == 'c' || strtolower($this->method) == 'd' ? true : false;
    }

    public function isWebOrder(){
        return strtolower($this->method) == 'w' ? true : false;
    }

    public function isOffline(){
        return !$this->online ? true : false;
    }

    public function isLegacy(){
        return $this->id <= 505009 ? true : false;
    }

    public function isRedo(){
        return !empty($this->reason_oid) ? true : false;
    }

    public function isReceivedAt(){
        return !empty($this->statusModel) && $this->statusModel->name == OrderStatus::STATUS_RECEIVED_AT_LAB ? true : false;
    }

    public function getPatient(){
        switch ($this->type) {
            case self::ORDER_RX:
                if(empty($this->rxOrder))
                    return '';
                return $this->rxOrder->first_name.' '.$this->rxOrder->last_name;
                break;
            default:
                return $this->patient;
                break;
        }
    }
    /**
     * @return mixed
     */
    public function rxOrderModel()
    {
        return $this->hasOne(RxOrders::class,'order_id','id');
    }

    public function isRx(){
        return $this->type == self::ORDER_RX ? true : false;
    }

    public function isStock(){
        return $this->type == self::ORDER_STOCK ? true : false;
    }

    public function isConsumables(){
        return $this->type == self::ORDER_CONSUMABLE ? true : false;
    }

    /**
     * @return mixed
     */
    public function statusModel()
    {
        return $this->hasOne(OrderStatus::class,'id','status');
    }

    /**
     * @return string
     */
    public function stockOrders() {
        return $this->hasMany(StockOrder::class);
    }

    /**
     * @return mixed
     */
    public function rxOrder()
    {
        return $this->hasOne(RxOrders::class);
    }

    /**
     * @return mixed
     */
    public static function getOrderTypeList()
    {
        return [
            self::ORDER_RX => 'Rx Lens',
            self::ORDER_STOCK => 'Stock',
            self::ORDER_CONSUMABLE => 'Consumable'
        ];
    }

    public function isSaved(){
        return $this->is_saved == 1 ? true : false;
    }


    /**
     * Creates new order entry
     * @param  \Illuminate\Http\Request  $request
     * @param  $type
     * @param  $price
     * @return \Illuminate\Http\Response
     */
    public static function newOrder($request, $type = null, $price = null) {
        return DB::transaction(function () use ($request,$type,$price) {
            if ($type == self::ORDER_STOCK) {
                // Price for both eye combinitions
                if ( isset($request["l_price"]) && isset($request["r_price"]) ) {
                    $price = $request["l_price"] + $request["r_price"];
                } else {
                    // Price for left eye combinitions
                    if(isset($request["l_price"])) {
                        $price = $request["l_price"];
                    // Price for right eye combinitions
                    } elseif (isset($request["r_price"])) {
                        $price = $request["r_price"];
                    }
                }
                $post = [
                    'user_id' => auth()->user()->getCustomerAdmin()->id,
                    'patient' => $request["patient"],
                    'type'    => self::ORDER_STOCK,
                    'status'  => $request['submit_type'] == OrderStatus::STATUS_SUBMITTED ? OrderStatus::getStatus(OrderStatus::STATUS_SUBMITTED)->id : OrderStatus::getStatus(OrderStatus::STATUS_SAVED)->id,
                    'price'   => $price
                ];
                if(isset($request['reason'])){
                    $post['reason_oid'] = $request['reason']['id'];
                    $post['reason_type'] = $request['reason']['reason'];
                    $post['reason_notes'] = $request['reason']['notes'];
                }
               return  parent::create($post);

            } elseif ($type == self::ORDER_CONSUMABLE) {
                $post = [
                    'user_id' => auth()->user()->getCustomerAdmin()->id,
                    'type'    => self::ORDER_CONSUMABLE,
                    'status'  => OrderStatus::getStatus(OrderStatus::STATUS_SUBMITTED)->id,
                    'price'   => $price,
                    'multiple'  =>  true
                ];
                if(isset($request['reason'])){
                    $post['reason_oid'] = $request['reason']['id'];
                    $post['reason_type'] = $request['reason']['reason'];
                    $post['reason_notes'] = $request['reason']['notes'];
                }
                return parent::create($post);
            } elseif ($type == self::ORDER_BARCODE) {

                return parent::create([
                    'user_id' => auth()->user()->getCustomerAdmin()->id,
                    'type'    => self::ORDER_STOCK,
                    'status'  => OrderStatus::getStatus(OrderStatus::STATUS_SUBMITTED)->id,
                    'price'   => $request->price
                ]);
            } elseif ($type == self::ORDER_RX) {
                $post = [
                    'user_id' => auth()->user()->getCustomerAdmin()->id,
                    'type'    => self::ORDER_RX,
                    'price'  => $request['data']['price'],
                    'patient' => $request['data']["first_name"].' '.$request['data']["last_name"],
                    'status'  => $request['submit_type'] == OrderStatus::STATUS_SUBMITTED ? OrderStatus::getStatus(OrderStatus::STATUS_SUBMITTED)->id : OrderStatus::getStatus(OrderStatus::STATUS_SAVED)->id,
                    'is_saved'  =>  $request['submit_type'] == OrderStatus::STATUS_SUBMITTED ? 0 : 1
                ];
                if(isset($request['reason'])){
                    $post['reason_oid'] = $request['reason']['id'];
                    $post['reason_type'] = $request['reason']['reason'];
                    $post['reason_notes'] = $request['reason']['notes'];
                }
                $order = null;
                if(!empty($request['data']['order_id'])){
                    $order = self::where('user_id',auth()->user()->getCustomerAdmin()->id)
                                    ->where('id',$request['data']['order_id'])
                                    ->first();
                    if(!empty($order)){
                        if($order->isSaved() && $post['is_saved'] == OrderStatus::STATUS_SUBMITTED){
                            $post['created_at'] = date('Y-m-d H:i:s');
                        }
                        $order->update($post);
                    }
                }else{
                    if($request['submit_type'] == OrderStatus::STATUS_SAVED)
                       $post['saved_at'] = date('Y-m-d H:i:s');
    
                    $order = parent::create($post);
                }
                return $order;
            }
        });
    }

    public function notifications()
    {
        return $this->hasMany(Notification::class, 'order_id', 'id');
    }

    public function search(Request $request, $currentPage, $recordsPerPage, $isSaved = false)
    {
        $post = $request->all();
        $user = auth()->user()->getCustomerAdmin();
        $currentUser = auth()->user();
        $isAdmin = $currentUser->isAdmin();
        $notificationRecipientId = $isAdmin ? 1 : $currentUser->id;

        // Optimize columns based on user type
        $columns = [
            'id', 'user_id', 'status', 'type', 'method', 'order_csv', 'invoice_no',
            'patient', 'created_at', 'eta', 'is_saved', 'tracking_no', 'shipping_carrier',
            'status_date_time', 'shipped_date', 'received_at_lab_date', 'online', 'reason_oid', 'multiple'
        ];

        // Always load essential relationships
        $relationships = [
            'user.profile',
            'statusModel',
            'rxOrder'
        ];

        // Optimized subquery approach with performance hints
        $model = Order::with($relationships)
            ->select($columns)
            ->selectRaw('(SELECT COUNT(*) FROM notifications WHERE notifications.order_id = orders.id AND notifications.has_read = 0 AND notifications.to = ?) as unread_notifications_count', [$notificationRecipientId])
            ->where('is_saved', $isSaved)
            ->where('status', '!=', 43);

        // Filter by status
        if (isset($post['status']) && !empty($post['status'])) {
            $model = $model->whereIn('status', $post['status']);
        }

        // Filter by user IDs
        if (isset($post['user_ids']) && !empty($post['user_ids'])) {
            $model = $model->whereIn('user_id', $post['user_ids']);
        }

        // Search by ID, invoice number, or patient
        if (!empty($post['search'])) {
            $search = $post['search'];
            // Check if search string starts with 'W' and remove it if present
            if (str_starts_with(strtoupper($search), 'W')) {
                $search = substr($search, 1);
            }
            $model = $model->where(function ($query) use ($search) {
                $query->where('id', $search)
                    ->orWhere('invoice_no', $search)
                    ->orWhere('order_csv', $search)
                    ->orWhere('patient', 'LIKE', '%' . $search . '%');
            });
        }

        // Filter by date range
        if (!empty($post['from_date'])) {
            $model = $model->whereDate('created_at', '>=', $post['from_date']);
        }
        if (!empty($post['to_date'])) {
            $model = $model->whereDate('created_at', '<=', $post['to_date']);
        }

        // Filter by customer admin/staff
        if ($user->isCustomerAdmin() || $user->isCustomerStaff()) {
            $model = $model->where('user_id', $user->getCustomerAdmin()->id);
        }

        // Sorting logic
        if (!empty($request->get('order'))) {
            $id = $request->get('order');
            $model = $model->orderByRaw("FIELD(id, ?) DESC", [$id])
                        ->orderBy('created_at', 'desc');
        } else {
            $model = $model->orderBy('created_at', 'desc');
        }

        // Apply pagination with optimized settings
        $model = $model->paginate($recordsPerPage, ['*'], 'page-no', $currentPage);

        return $model;
    }

    public function makeCopy(){
        if($this->isRx()){
            $orderModel = $this->replicate();
            $orderModel->save();
            if(!empty($orderModel)){
                $rxOrder = $this->rxOrderModel->replicate();
                $rxOrder->order_id = $orderModel->id;
                $rxOrder->save();
                return $orderModel;
            }
        }
        return false;
    }

    public function uploadPDFToDropbox(){
        if(!$this->isSaved()){
            $order = Order::where('id', $this->id)->first();
            // Fire the event for asynchronous processing via listener
            event(new \App\Events\OrderPlaced($order));
        }
    }

    public function saveOffline($params,$type){
        $userModal = User::where('ac_code',$params['ac_code'])->first();
        $statusModal = OrderStatus::where('title',$params['status'])->first();
        if(!empty($userModal) && !empty($statusModal)){
            if($type == 'Rx'){
                $orderModal = self::create([
                    'user_id' => $userModal->id,
                    'patient' => $params['patient'],
                    'tray' => $params['tray'],
                    'online' => $params['online'],
                    'invoice_no' => $params['invoice_no'],
                    'shipping_carrier' => $params['shipping_carrier'],
                    'tracking_no' => $params['tracking_no'],
                    'is_uploaded' => $params['online'],
                    'type'    => Order::ORDER_RX,
                    'status'  => $statusModal->id,
                    'price'   => 0,
                    'created_at'    =>  $params['created_at'],
                    'updated_at'    =>  $params['created_at'],
                    'order_csv' =>  $params['order_csv']
                ]);
                if(!empty($orderModal)){
                    $metaData = [
                        'price' =>  0,
                        'first_name'    =>  $params['patient'],
                        'last_name'     =>  '',
                        'special_instructions'  =>  '',
                        'lens'  =>   [
                            'lens_type' =>  '',
                            'lens_type2'    =>  '',
                            'sub_type'    =>  '',
                            'lens_manufacturer'    =>  '',
                            'base_curve'    =>  '',
                            'lens_material'    =>  '',
                            'lens_filter'    =>  '',
                            'lens_colour'    =>  '',
                            'coating_made'    =>  '',
                            'coating_name'    =>  '',
                        ],
                        'rx'    =>  [
                            'distance'  =>  [
                                'right_sph' =>  '',
                                'right_cyl' =>  '',
                                'right_axis' =>  '',
                                'right_oc' =>  '',
                                'left_sph' =>  '',
                                'left_cyl' =>  '',
                                'left_axis' =>  '',
                                'left_oc' =>  '',
                            ],
                            'prism' =>  [
                                'right_prism1' => '',
                                'right_base1'   =>  '',
                                'right_prism2' => '',
                                'right_base2'   =>  '',
                                'left_prism1' => '',
                                'left_base1'   =>  '',
                                'left_prism2' => '',
                                'left_base2'   =>  '',
                            ],
                            'add_for_reading'   =>  [
                                'right_add' =>  '',
                                'right_boc' =>  '',
                                'right_sh' =>  '',
                                'right_intermediate_strength' =>  '',
                                'right_boc' =>  '',
                                'left_add' =>  '',
                                'left_boc' =>  '',
                                'left_sh' =>  '',
                                'left_intermediate_strength' =>  '',
                                'left_boc' =>  ''
                            ],
                            'extended_info'=>[
                                'right_pd'  =>  '',
                                'right_intermediate_pd'  =>  '',
                                'right_npd'  =>  '',
                                'right_base_curve'  =>  '',
                                'left_pd'  =>  '',
                                'left_intermediate_pd'  =>  '',
                                'left_npd'  =>  '',
                                'left_base_curve'  =>  '',
                            ],
                            'special_info'=>[
                                'pento' =>  '',
                                'wrap'  =>  '',
                                'tilt'  =>  ''
                            ]
                        ],
                        'frame' =>  [
                            'job_type'  =>  '',
                            'frame_type'    =>  '',
                            'invoice_order_no'  =>  '',
                            'ft_other'  =>  '',
                            'specify_thickness' =>  '',
                            'safety_stamp'  =>  '',
                            'bevel_type'    =>  '',
                            'edge_polish'   =>  '',
                            'frame_info'    =>  [
                                'brand' =>  '',
                                'model' =>  '',
                                'colour' =>  '',
                                'pof' =>  ''
                            ],
                            'frame_size'    =>  [
                                'a' =>  '',
                                'b' =>  '',
                                'ed' =>  '',
                                'dbl' =>  '',
                                'shape' =>  '',
                                'ed_or_shape'   =>  ''
                            ],
                            'modify_shape'  =>  [
                                'a_increased_by'    =>  '',
                                'a_shape_options'   =>  '',
                                'b_increased_by'    =>  '',
                                'b_shape_options'   =>  ''
                            ],
                            'specify_thickness' =>  [
                                'ct_or_et'  =>  ''
                            ]
                        ],
                        'additional_items'  =>  [
                            'add_clip_on'   =>  '',
                            'add_repairs'   =>  '',
                            'clip_on_list'  =>  '',
                            'repair_description'    =>  ''
                        ],
                        'is_tinting'  =>  '',
                        'tint_option'  =>  '',
                        'tint_color_sample'  =>  '',
                        'tinting'  =>  '',
                        'edging'  =>  '',
                        'coat_id'  =>  '',
                        'coat_price'  =>  '',
                        'coat_values'  =>  '',
                        'reader'  =>  '',
                        'att_values'  =>  '',
                        'group_values'  =>  '',
                        'tint_title'  =>  '',
                        'is_multipair'  =>  '',
                        'multipair_ref' =>  ''
                    ];
                    RxOrders::create([
                        'order_id' => $orderModal->id,
                        'first_name'  => $params['patient'],
                        'last_name'  => '',
                        'data'  => json_encode($metaData),
                    ]);
                    return $orderModal;
                }

            }elseif($type == 'Stock'){
                $orderModal = self::create([
                    'user_id' => $userModal->id,
                    'patient' => $params['patient'],
                    'tray' => $params['tray'],
                    'online' => $params['online'],
                    'invoice_no' => $params['invoice_no'],
                    'shipping_carrier' => $params['shipping_carrier'],
                    'tracking_no' => $params['tracking_no'],
                    'is_uploaded' => $params['online'],
                    'type'    => Order::ORDER_STOCK,
                    'status'  => $statusModal->id,
                    'price'   => 0,
                    'created_at'    =>  $params['created_at'],
                    'updated_at'    =>  $params['created_at'],
                    'order_csv' =>  $params['order_csv']
                ]);
                if(!empty($orderModal)){
                    $postData['type'] = StockOrder::LEFT_EYE;
                    $postData['order_id'] = $orderModal->id;
                    StockOrder::create($postData);
                    return $orderModal;
                }
            }
        }
        return null;
    }

    public function updateOrderStatus($data, $online = null)
    {
        $statusModal = OrderStatus::where('title', $data['Status'])->first();

        $payloads = [
            'invoice_no'    => $data['Invoice_Number'],
            'patient'       => $data['Patient'] ?? null,
            'tray'          => $data['Tray'] ?? null,
            'shipping_carrier' => $data['Ship_Via'] ?? null,
            'tracking_no'   => $data['Tracking_Ref'] ?? null,
        ];

        // Set the `online` field if provided
        if ($online !== null) {
            $payloads['online'] = $online;
        }

        // Link the status ID if it exists
        if (!empty($statusModal)) {
            $payloads['status'] = $statusModal->id;
        }

        // Set the `order_csv` field for offline orders
        if ($online !== null && !$online) {
            $payloads['order_csv'] = $data['Order_Number'] ?? null;
        }

        // Parse and set the `status_date_time` field
        if (!empty($data['Status_Date']) && strpos($data['Status_Date'], "/") !== false) {
            $parts = explode('/', $data['Status_Date']);
            $payloads['status_date_time'] = convert_date_time_utc(
                date('Y-m-d H:i:s', strtotime($parts[2] . "-" . $parts[0] . "-" . $parts[1] . ' ' . ($data['Status_Time'] ?? '00:00:00')))
            );
        }

        // Parse and set the `eta` field
        if (!empty($data['ETA'])) {
            $payloads['eta'] = date('Y-m-d', strtotime($data['ETA']));
        }else{
            $payloads['eta'] = null;
        }

        // Extract the method from the first character of the order number, if not numeric
        if (!is_numeric(substr($data['Order_Number'], 0, 1))) {
            $payloads['method'] = substr($data['Order_Number'], 0, 1);
        }

        $this->update($payloads);
    }

    public static function getReasonsArrayList(){
        return [
            "Dr's Rx Change (951)",
            "Non-Adapt (953)",
            "Warranty (965)",
            "Received Defective Lenses (951)",
            "Customer Edging Mistake (958)",
            "Customer Ordering Mistake (959)",
            "Lab Mistake (960)"
        ];
    }

    public static function isEditableWithReason($value){
        $response = false;
        $array = ["Dr's Rx Change (951)",'Non-Adapt (953)','Received Defective Lenses (951)','Customer Ordering Mistake (959)'];
        if(in_array($value, $array))
            $response = true;

        return $response;
    }
}
