<?php

namespace App\Models\Backend;

use DB;
use App\Models\Backend\StockOrder;
use App\Models\Auth\User;
use Illuminate\Database\Eloquent\Model;

class PhoneLog extends Model
{
    const ORDER_LOG = 'orders';
    const SIMPLE_LOG = 'phone';
    const TYPE_NOTE = 'notes';
    const TYPE_REPLY = 'reply';
    const TYPE_QUESTION = 'questions';
    const TYPE_PHONE_LOG = 'phone_log';
    const TYPE_REQUEST_CHANGE = 'request_change';
    const TYPE_ORDER_CANCELLATION = 'order_cancellation';

    protected $table = "phone_logs";

    protected $fillable = [
        'id',
        'type',
        'user_id',
        'order_id',
        'user_to',
        'replied',
        'description',
    ];

    /**
     * @return mixed
     */
    public function user()
    {
        return $this->hasOne(User::class,'id','user_id');
    }

    /**
     * @return mixed
     */
    public function order()
    {
        // return $this->hasOne(Order::class,'id','order_id');
        return $this->belongsTo(Order::class);
    }

    /**
     * @return mixed
     */
    public function userTo()
    {
        return $this->hasOne(User::class,'id','user_to');
    }

    /**
     * @return bool
     */
    public function isNote()
    {
        return $this->type == self::TYPE_NOTE ? true : false;
    }

    /**
     * @return bool
     */
    public function isQuestion()
    {
        return $this->type == self::TYPE_QUESTION ? true : false;
    }

    /**
     * @return bool
     */
    public function isRequestChange()
    {
        return $this->type == self::TYPE_REQUEST_CHANGE ? true : false;
    }

    /**
     * @return bool
     */
    public function isRequestCancellation()
    {
        return $this->type == self::TYPE_ORDER_CANCELLATION ? true : false;
    }

     /**
     * @return bool
     */
    public function isPhoneLog()
    {
        return $this->type == self::TYPE_PHONE_LOG ? true : false;
    }

    public function replies()
    {
        return $this->hasMany(PhoneLog::class, 'replied');
    }

    public function parent()
    {
        return $this->belongsTo(PhoneLog::class, 'replied');
    }

    /**
     * @return bool
     */
    // public function replyModel()
    // {
    //     // return PhoneLog::where('replied',$this->id)->first();
    //     return PhoneLog::where('replied',$this->id)->get()->toArray();
    // }

    public function replyModel()
    {
        return [
            'logs' => PhoneLog::with('user')->where('replied', $this->id)->get()->toArray(),
            'user' => auth()->user() ? auth()->user()->only(['id', 'first_name', 'email', 'avatar_location']) : null,
        ];
    }

    /**
     * @return bool
     */
    public function isReply()
    {
        return $this->replied != null ? true : false;
    }

    /**
     * @return mixed
     */
    public function getTypeText()
    {
        switch($this->type){
            case self::TYPE_QUESTION:
                return 'Question';
            break;
            case self::TYPE_PHONE_LOG:
                return 'Phone Log';
            break;
            case self::TYPE_NOTE:
                return 'Phone Log';
            break;
            default:
                return ucwords($this->type);
            break;
        }
    }

    /**
     * @return mixed
     */
    public function getIcon()
    {
        switch($this->type){
            case self::TYPE_QUESTION:
                return '<i class="fa fa-question-circle" aria-hidden="true"></i>';
            break;
            case self::TYPE_PHONE_LOG:
                return '<i class="fa fa-phone" aria-hidden="true"></i>';
            break;
            case self::TYPE_REQUEST_CHANGE:
                return '<i class="fa fa-pencil" aria-hidden="true"></i>';
            break;
            case self::TYPE_ORDER_CANCELLATION:
                return '<i class="fa fa-times" aria-hidden="true"></i>';
            break;
            default:
                return '<i class="fa fa-sticky-note" aria-hidden="true"></i>';
            break;
        }

    }

    public static function getNotifyIcon($type)
    {
         switch($type){
            case self::TYPE_QUESTION:
                return '<span class="badge bg-info"><i class="fa fa-question-circle" aria-hidden="true"></i> Order Enquiry</span>';
            case self::TYPE_PHONE_LOG:
                return '<span class="badge bg-info"><i class="fa fa-phone" aria-hidden="true"></i> Phone Enquiry</span>';
            case self::TYPE_REQUEST_CHANGE:
                return '<span class="badge bg-warning"><i class="fa fa-pencil" aria-hidden="true"></i> Order Change</span>';
            case self::TYPE_ORDER_CANCELLATION:
                return '<span class="badge bg-danger"><i class="fa fa-times" aria-hidden="true"></i> Order Cancellation</span>';
            case self::TYPE_REPLY:
                return '<span class="badge bg-info"><i class="fa fa-reply fa-2xl" aria-hidden="true"></i> Replied</span>';
            default:
                return '<span class="badge bg-info"><i class="fa fa-question-circle" aria-hidden="true"></i> Order Enquiry</span>';
            break;
        }

    }
    /**
     * @return mixed
     */
    public function getUserImage()
    {
        $user = $this->user;
        if($user->avatar_type != 'gravatar' && !empty($user->avatar_location))
            return $user->picture;

        return url('public/favicon.ico');
    }

    /**
     * @return mixed
     */
    public function getProfileImage()
    {
        $user = $this->user;
        if($user->avatar_type != 'gravatar' && !empty($user->avatar_location))
            return $user->picture;

        return null;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return \Illuminate\Http\Response
     * @param  Request  $request
     */
 
    public static function store($data) {
        return parent::create([
            'user_id' => auth()->user()->id,
            'order_id' => isset($data["order_id"]) ? $data["order_id"] : null,
            'description' => $data["description"],
            'type' => $data["type"],
            'user_to'   =>  isset($data["user_to"]) ? $data["user_to"] : null,
            'replied'   =>  isset($data["replied"]) ? $data["replied"] : null
        ]);
    }

    /**
     * Get logs by customers.
     *
     * @return \Illuminate\Http\Response
     * @param  Request  $request
     */
    public static function getPhonelogs($type,$orderId) {
        if ($type == self::ORDER_LOG){
            if (auth()->user()->isCustomerAdmin() || auth()->user()->isCustomerStaff()) {
                return self::where('replied',null)
                    ->where('type','!=',self::TYPE_NOTE)
                    ->where('order_id',$orderId)
                    ->orderBy('updated_at','desc')
                    ->get();
            } else {
                return self::where('replied',null)
                    ->where('order_id',$orderId)
                    ->orderBy('updated_at','desc')
                    ->get();
            //     return  self::where('order_id',$orderId)
            //         ->orderBy('updated_at','desc')
            //         ->get();
            }
        } elseif ($type == self::SIMPLE_LOG) {
            return self::where('order_id',null)
                ->where('user_to',$orderId)
                ->orderBy('updated_at','desc')
                ->get();
        }
    }

    public function isDeleteTimeout() {
        // return now()->diffInMinutes($this->created_at) <= 300;
        return now()->diffInSeconds($this->created_at) <= 30;
    }
}
