<?php

namespace App\Repositories\Backend\Auth;

use App\Models\Auth\User;
use App\Models\Auth\Role;
use Illuminate\Support\Facades\DB;
use App\Models\Backend\UserSettings;
use App\Models\Backend\UserProfile;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use App\Models\Backend\CustomerLinkStaff;
use App\Events\Backend\Auth\User\UserCreated;
use App\Events\Backend\Auth\User\UserUpdated;
use App\Events\Backend\Auth\User\UserRestored;
use App\Events\Backend\Auth\User\UserConfirmed;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Events\Backend\Auth\User\UserDeactivated;
use App\Events\Backend\Auth\User\UserReactivated;
use App\Events\Backend\Auth\User\UserUnconfirmed;
use App\Events\Backend\Auth\User\UserPasswordChanged;
use App\Notifications\Backend\Auth\UserAccountActive;
use App\Events\Backend\Auth\User\UserPermanentlyDeleted;
use App\Notifications\Frontend\Auth\UserNeedsConfirmation;

/**
 * Class UserRepository.
 */
class UserRepository extends BaseRepository
{
    /**
     * @return string
     */
    public function model()
    {
        return User::class;
    }

    /**
     * @return mixed
     */
    public function getUnconfirmedCount() : int
    {
        return $this->model
            ->where('confirmed', false)
            ->count();
    }

    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return mixed
     */
    public function getActivePaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc',$params = [], $pageNo = 1) : LengthAwarePaginator
    {
        $user = auth()->user();
        if ($user->isCustomerAdmin()) {
            $users = $user->getCustomerStaff();
            $model = $this->model->whereIN('id',$users);

            return $model
                ->with('roles', 'permissions', 'providers')
                ->active()
                ->orderBy($orderBy, $sort)
                ->paginate($paged, ['*'], 'page', $pageNo);
        } else {
            $users = $user->getUsersList();
            $model = $this->model->leftJoin('user_profile', 'user_profile.user_id', '=', 'users.id')->join('model_has_roles', 'model_has_roles.model_id', '=', 'users.id')
            ->select('users.*')
            ->with('roles', 'permissions', 'providers')
            ->whereIN('users.id',$users);

            if(isset($params['search']) && !empty($params['search'])){
                $search = $params['search'];
                $model = $model->where(function($query) use ($search) {
                    $query->orWhere('ac_code', 'LIKE', '%' . $search . '%')
                        ->orWhere('first_name', 'LIKE', '%' . $search . '%')
                        ->orWhere('last_name', 'LIKE', '%' . $search . '%')
                        ->orWhere('user_profile.business_name', 'LIKE', '%' . $search . '%');
                });
            }
            if(isset($params['status']) && !empty($params['status'])){
                $model = $model->whereIN('role_id',$params['status']);
            }

            $model = $model//->active()
            ->orderBy($orderBy, $sort)
            ->paginate($paged, ['*'], 'page', $pageNo);

            return $model;
        }
    }

    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return LengthAwarePaginator
     */
    public function getInactivePaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc') : LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->active(false)
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }

    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return LengthAwarePaginator
     */
    public function getDeletedPaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc') : LengthAwarePaginator
    {
        return $this->model
            ->with('roles', 'permissions', 'providers')
            ->onlyTrashed()
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
    }

    /**
     * @param array $data
     *
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function create(array $data) : User
    {
        return DB::transaction(function () use ($data) {
            
            $user = parent::create([
                'first_name'        =>  $data['first_name'],
                'last_name'         =>  $data['last_name'],
                'email'             =>  $data['email'],
                'password'          =>  $data['password'],
                'active'            =>  1,
                'confirmation_code' =>  md5(uniqid(mt_rand(), true)),
                'confirmed'         =>  isset($data['confirmed']) && $data['confirmed'] === '1',
                'ac_code'           => (!empty($data['ac_code'])) ? $data['ac_code'] : ''
            ]);
            if ($user) {
                // Add selected roles/permissions
                $user->syncRoles((int)$data['roles']);

                $user->updateProfile($data);

                //Send confirmation email if requested and account approval is off
                // if ($user->confirmed === false && isset($data['confirmation_email']) && ! config('access.users.requires_approval')) {
                //     $user->notify(new UserNeedsConfirmation($user->confirmation_code));
                // }

                // event(new UserCreated($user));

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.create_error'));
        });
    }

    /**
     * @param User          $user

     * @param array $data
     *
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function update(User $user, array $data) : User
    {
        $this->checkUserByEmail($user, $data['email']);
        return DB::transaction(function () use ($user, $data) {
            $params = [
                'first_name'    =>  $data['first_name'],
                'last_name'     =>  $data['last_name'],
                'email'         =>  $data['email']
            ];
            if(isset($data['active']))
                $params['active'] = $data['active'];

            $user->update($params);

            $user->updateProfile($data);
            //Address is not current address so they need to reconfirm
            if ($user->email !== $data['email']) {
                //Emails have to be unique
                if ($this->getByColumn($data['email'], 'email')) {
                    throw new GeneralException(__('exceptions.frontend.auth.email_taken'));
                }
                $user->email = $data['email'];
                $updated = $user->save();
                // Send the new confirmation e-mail
                return [
                    'success' => $updated,
                    'email_changed' => true,
                ];
            }
            event(new UserUpdated($user));

            return $user;
        });
    }
    
    /**
     * @param User          $user

     * @param array $data
     *
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function updateSettings(User $user, array $data)
    {
        return DB::transaction(function () use ($user, $data) {
            $model = UserSettings::where('user_id',$user->id)->first();

            if(empty($model)){
                $data['user_id'] = $user->id;
                UserSettings::create($data);
            }
            else
                $model->update($data);

            return $user;
        });
    }

    /**
     * @param User          $user

     * @param array $data
     *
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function updateAvatar(User $user, array $data, $image = false) : User
    {
        return DB::transaction(function () use ($user, $data, $image) {
            // Upload profile image if necessary
            if ($image) {
                $user->avatar_type  = $data['avatar_type'];
                $user->avatar_location = $image->store('/avatars', 'public');
                $user->update();
            } else {
                // No image being passed
                if (!empty($data['avatar_type']) && $data['avatar_type'] === 'storage') {
                    // If there is no existing image
                    if (auth()->user()->avatar_location === '') {
                        throw new GeneralException('You must supply a profile image.');
                    }
                }
            }
            return $user;
        });
    }

    /**
     * @param User $user
     * @param      $input
     *
     * @throws GeneralException
     * @return User
     */
    public function updatePassword(User $user, $input) : User
    {
        if ($user->update(['password' => $input['password']])) {
            event(new UserPasswordChanged($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.update_password_error'));
    }

    /**
     * @param User $user
     * @param      $status
     *
     * @throws GeneralException
     * @return User
     */
    public function mark(User $user, $status) : User
    {
        if ($status === 0 && auth()->id() === $user->id) {
            throw new GeneralException(__('exceptions.backend.access.users.cant_deactivate_self'));
        }

        $user->active = $status;

        switch ($status) {
            case 0:
                event(new UserDeactivated($user));
            break;
            case 1:
                event(new UserReactivated($user));
            break;
        }

        if ($user->save()) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.mark_error'));
    }

    /**
     * @param User $user
     *
     * @throws GeneralException
     * @return User
     */
    public function confirm(User $user) : User
    {
        if ($user->confirmed) {
            throw new GeneralException(__('exceptions.backend.access.users.already_confirmed'));
        }

        $user->confirmed = true;
        $confirmed = $user->save();

        if ($confirmed) {
            event(new UserConfirmed($user));

            // Let user know their account was approved
            if (config('access.users.requires_approval')) {
                $user->notify(new UserAccountActive);
            }

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.cant_confirm'));
    }

    /**
     * @param User $user
     *
     * @throws GeneralException
     * @return User
     */
    public function unconfirm(User $user) : User
    {
        if (! $user->confirmed) {
            throw new GeneralException(__('exceptions.backend.access.users.not_confirmed'));
        }

        if ($user->id === 1) {
            // Cant un-confirm admin
            throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm_admin'));
        }

        if ($user->id === auth()->id()) {
            // Cant un-confirm self
            throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm_self'));
        }

        $user->confirmed = false;
        $unconfirmed = $user->save();

        if ($unconfirmed) {
            event(new UserUnconfirmed($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.cant_unconfirm'));
    }

    /**
     * @param User $user
     *
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function forceDelete(User $user) : User
    {
        // if ($user->deleted_at === null) {
        //     throw new GeneralException(__('exceptions.backend.access.users.delete_first'));
        // }

        return DB::transaction(function () use ($user) {
            // Delete associated relationships
            $user->passwordHistories()->delete();
            $user->providers()->delete();

            if ($user->forceDelete()) {
                event(new UserPermanentlyDeleted($user));

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.delete_error'));
        });
    }

    /**
     * @param User $user
     *
     * @throws GeneralException
     * @return User
     */
    public function restore(User $user) : User
    {
        if ($user->deleted_at === null) {
            throw new GeneralException(__('exceptions.backend.access.users.cant_restore'));
        }

        if ($user->restore()) {
            event(new UserRestored($user));

            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.restore_error'));
    }

    /**
     * @param User $user
     * @param      $email
     *
     * @throws GeneralException
     */
    protected function checkUserByEmail(User $user, $email)
    {
        // Figure out if email is not the same and check to see if email exists
        if ($user->email !== $email && $this->model->where('email', '=', $email)->first()) {
            throw new GeneralException(trans('exceptions.backend.access.users.email_error'));
        }
    }

    /**
     * @param User   $user
     * @return mixed
     */
    public function getCustomerStaff($user) {
        $staff = CustomerLinkStaff::where('customer_id', $user->id)->get();
        $users = $staff->pluck('user_id');
        $model = User::whereIn('id',$users)->get();
        return $model;      
    }

    /**
     * @param array $data
     *
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function sync(array $data): User
    {
        return DB::transaction(function () use ($data) {
            $user = User::where('ac_code', $data['ac_code'])->first();
            $sendEmailFlag = false;

            if (empty($user)) {
                $user = parent::create([
                    'first_name'        => $data['first_name'],
                    'last_name'         => $data['last_name'],
                    'email'             => $data['email'],
                    'password'          => $data['password'],
                    'active'            => 1,
                    'confirmation_code' => md5(uniqid(mt_rand(), true)),
                    'confirmed'         => 1,
                    'ac_code'           => $data['ac_code'],
                    'created_at'        => $data['created_at'],
                    'updated_at'        => $data['updated_at'],
                ]);

                $role = Role::findByType(config('access.users.customer_admin'));
                $user->syncRoles($role->id);
                $sendEmailFlag = true;
            } else {
                unset($data['password']);
                $user->first_name = $data['first_name'];
                $user->last_name = $data['last_name'];
                $user->created_at = $data['created_at'];
                $user->update($data);
            }

            $user->updateProfile($data);

            // Move email sending OUTSIDE the transaction
            // DB::afterCommit(function () use ($user, $data, $sendEmailFlag) {
                if ($sendEmailFlag) {
                    $user->sendWelcomeEmail($data['password']);
                }
            // });

            return $user;
        });
    }
}
