<?php

namespace App\Repositories\Frontend\Auth;

use App\Models\Auth\Role;
use App\Models\Auth\User;
use Illuminate\Http\UploadedFile;
use App\Models\Backend\UserSettings;
use App\Models\Auth\SocialAccount;
use Illuminate\Support\Facades\DB;
use App\Exceptions\GeneralException;
use App\Repositories\BaseRepository;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use App\Events\Frontend\Auth\UserConfirmed;
use App\Events\Frontend\Auth\UserProviderRegistered;
use App\Notifications\Frontend\Auth\UserNeedsConfirmation;
use App\Models\Backend\CustomerLinkStaff;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Events\Backend\Auth\User\UserCreated;
use App\Events\Backend\Auth\User\UserUpdated;
use App\Events\Backend\Auth\User\UserPermanentlyDeleted;

/**
 * Class UserRepository.
 */
class UserRepository extends BaseRepository
{
    /**
     * @return string
     */
    public function model()
    {
        return User::class;
    }

    /**
     * @param $token
     *
     * @return bool|\Illuminate\Database\Eloquent\Model
     */
    public function findByPasswordResetToken($token)
    {
        foreach (DB::table(config('auth.passwords.users.table'))->get() as $row) {
            if (password_verify($token, $row->token)) {
                return $this->getByColumn($row->email, 'email');
            }
        }

        return false;
    }

    /**
     * @param $uuid
     *
     * @throws GeneralException
     * @return mixed
     */
    public function findByUuid($uuid)
    {
        $user = $this->model
            ->uuid($uuid)
            ->first();

        if ($user instanceof $this->model) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.not_found'));
    }

    /**
     * @param $code
     *
     * @throws GeneralException
     * @return mixed
     */
    public function findByConfirmationCode($code)
    {
        $user = $this->model
            ->where('confirmation_code', $code)
            ->first();

        if ($user instanceof $this->model) {
            return $user;
        }

        throw new GeneralException(__('exceptions.backend.access.users.not_found'));
    }

    /**
     * @param array $data
     *
     * @throws \Exception
     * @throws \Throwable
     * @return \Illuminate\Database\Eloquent\Model|mixed
     */
    public function create(array $data)
    {
        return DB::transaction(function () use ($data) {
            $user = parent::create([
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'email' => $data['roles'] == 5 ? null : $data['email'],
                'confirmation_code' => md5(uniqid(mt_rand(), true)),
                'active' => true,
                'password' => $data['password'],
                'confirmed' => isset($data['confirmed']) && $data['confirmed'] === '1',
                'ac_code' => $data['roles'] == Role::findByType(config('access.users.customer_staff'))->id ? $this->acCode() : $data['ac_code'],
            ]);

            if ($user) {
                // Add the default site role to the new user
                $user->assignRole(config('access.users.customer_staff'));
                CustomerLinkStaff::create([
                    'user_id' => $user->id,
                    'customer_id' => auth()->user()->id
                ]);
            }

            /*
             * If users have to confirm their email and this is not a social account,
             * and the account does not require admin approval
             * send the confirmation email
             *
             * If this is a social account they are confirmed through the social provider by default
             */
            if (config('access.users.confirm_email')) {
                // Pretty much only if account approval is off, confirm email is on, and this isn't a social account.
                $user->notify(new UserNeedsConfirmation($user->confirmation_code));
            }

            // Return the user object
            return $user;
        });
    }

    /**
     * @param       $id
     * @param array $input
     * @param bool|UploadedFile  $image
     *
     * @throws GeneralException
     * @return array|bool
     */
    public function update(User $user, array $data) : User
    {
        if(isset($data['email']))
            $this->checkUserByEmail($user, $data['email']);

        return DB::transaction(function () use ($user, $data) {
            $user->update([
                'first_name'    =>  $data['first_name'],
                'last_name'     =>  $data['last_name'],
                // 'email'         =>  $data['email']
            ]);
            if(isset($data['email']))
                $params['email'] = $data['email'];
            
            $user->updateProfile($data);
            //Address is not current address so they need to reconfirm
            if (isset($data['email']) && $user->email !== $data['email']) {
                //Emails have to be unique
                if ($this->getByColumn($data['email'], 'email')) {
                    throw new GeneralException(__('exceptions.frontend.auth.email_taken'));
                }
                $user->email = $data['email'];
                $updated = $user->save();
                
                // Send the new confirmation e-mail
                return [
                    'success' => $updated,
                    'email_changed' => true,
                ];
            }
            event(new UserUpdated($user));

            return $user;
        });
    }

    /**
     * @param User          $user

     * @param array $data
     *
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function updateSettings(User $user, array $data)
    {
        return DB::transaction(function () use ($user, $data) {
            $data['order_placed'] = isset($data['order_placed']) && $data['order_placed'] == 'on' ? true : false;
            $data['order_shipped'] = isset($data['order_shipped']) && $data['order_shipped'] == 'on' ? true : false;
            $data['exceptions'] = isset($data['exceptions']) && $data['exceptions'] == 'on' ? true : false;
            $model = UserSettings::where('user_id',$user->id)->first();
            // $defaultNtf = UserSettings::getNotifications();
            // if(!empty($data['n_email'])){
            //     foreach ($defaultNtf as $key => $value) {
            //         if(!isset($data['n_email'][$key]))
            //             $data['n_email'][$key] = 0;
            //     }
            // }else
            //     $data['n_email'] = $defaultNtf;
                
            // if(!empty($data['n_fax'])){
            //     foreach ($defaultNtf as $key => $value) {
            //         if(!isset($data['n_fax'][$key]))
            //             $data['n_fax'][$key] = 0;
            //     }
            // }else
            //     $data['n_fax'] = $defaultNtf;

            // $data['n_email'] = json_encode($data['n_email']);
            // $data['n_fax'] = json_encode($data['n_fax']);

            if(empty($model)){
                $data['user_id'] = $user->id;
                UserSettings::create($data);
            }
            else
                $model->update($data);

            return $user;
        });
    }

    /**
     * @param User          $user

     * @param array $data
     *
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function updateAvatar(User $user, array $data, $image = false) : User
    {
        return DB::transaction(function () use ($user, $data, $image) {
            // Upload profile image if necessary
            if ($image) {
                $user->avatar_type  = $data['avatar_type'];
                $user->avatar_location = $image->store('/avatars', 'public');
                $user->update();
            } else {
                $user->removePicture();
                $user->avatar_type  = 'gravatar';
                $user->avatar_location = '';
                $user->update();
                // No image being passed
                // if (!empty($data['avatar_type']) && $data['avatar_type'] === 'storage') {
                //     // If there is no existing image
                //     if (auth()->user()->avatar_location === '') {
                //         throw new GeneralException('You must supply a profile image.');
                //     }
                // }
            }
            return $user;
        });
    }

    /**
     * @param      $input
     * @param bool $expired
     *
     * @throws GeneralException
     * @return bool
     */
    public function updatePassword($input, User $user = null, $expired = false)
    {
        if ($user) {
            if ($user->update(['password' => $input['password']])) {
    
                return $user;
            }
    
            throw new GeneralException(__('exceptions.backend.access.users.update_password_error'));            
        }

        $currentUser = $this->getById(auth()->id());

        if ($input['password'] == $input['password_confirmation']) {
            if ($currentUser->isTempPass() || Hash::check($input['old_password'], $currentUser->password)) {

                if ($expired) {
                    $currentUser->password_changed_at = now()->toDateTimeString();
                }
                $currentUser->update(['password' => $input['password']]);
                return true;
            }
            throw new GeneralException('You entered an incorrect current password');
        }
        throw new GeneralException(__('exceptions.frontend.auth.password.password_mismatch'));
    }

    /**
     * @param $code
     *
     * @throws GeneralException
     * @return bool
     */
    public function confirm($code)
    {
        $user = $this->findByConfirmationCode($code);

        if ($user->confirmed === true) {
            throw new GeneralException(__('exceptions.frontend.auth.confirmation.already_confirmed'));
        }

        if ($user->confirmation_code === $code) {
            $user->confirmed = true;

            event(new UserConfirmed($user));

            return $user->save();
        }

        throw new GeneralException(__('exceptions.frontend.auth.confirmation.mismatch'));
    }

    /**
     * @param $data
     * @param $provider
     *
     * @throws GeneralException
     * @return mixed
     */
    public function findOrCreateProvider($data, $provider)
    {
        // User email may not provided.
        $user_email = $data->email ?: "{$data->id}@{$provider}.com";

        // Check to see if there is a user with this email first.
        $user = $this->getByColumn($user_email, 'email');

        /*
         * If the user does not exist create them
         * The true flag indicate that it is a social account
         * Which triggers the script to use some default values in the create method
         */
        if (! $user) {
            // Registration is not enabled
            if (! config('access.registration')) {
                throw new GeneralException(__('exceptions.frontend.auth.registration_disabled'));
            }

            // Get users first name and last name from their full name
            $nameParts = $this->getNameParts($data->getName());

            $user = parent::create([
                'first_name' => $nameParts['first_name'],
                'last_name' => $nameParts['last_name'],
                'email' => $user_email,
                'active' => true,
                'confirmed' => true,
                'password' => null,
                'avatar_type' => $provider,
            ]);

            if ($user) {
                // Add the default site role to the new user
                $user->assignRole(config('access.users.default_role'));
            }

            event(new UserProviderRegistered($user));
        }

        // See if the user has logged in with this social account before
        if (! $user->hasProvider($provider)) {
            // Gather the provider data for saving and associate it with the user
            $user->providers()->save(new SocialAccount([
                'provider' => $provider,
                'provider_id' => $data->id,
                'token' => $data->token,
                'avatar' => $data->avatar,
            ]));
        } else {
            // Update the users information, token and avatar can be updated.
            $user->providers()->update([
                'token' => $data->token,
                'avatar' => $data->avatar,
            ]);

            $user->avatar_type = $provider;
            $user->update();
        }

        // Return the user object
        return $user;
    }

    /**
     * @param int    $paged
     * @param string $orderBy
     * @param string $sort
     *
     * @return mixed
     */
    public function getActivePaginated($paged = 25, $orderBy = 'created_at', $sort = 'desc') : LengthAwarePaginator
    {
        $user = auth()->user();
        $response = [];
        if ($user->isCustomerAdmin()) {
            $users = $user->getCustomerStaff();
            $model = $this->model->whereIN('id',$users);

            $response = $model
                ->with('roles', 'permissions', 'providers')
                // ->active()
                ->orderBy($orderBy, $sort)
                ->paginate($paged);
        } else {
            $users = $user->getUsersList();
            $response = $this->model
            ->with('roles', 'permissions', 'providers')
            ->whereIN('users.id',$users)
            ->active()
            ->orderBy($orderBy, $sort)
            ->paginate($paged);
        }
        return $response;
    }

    /**
     * @param $fullName
     *
     * @return array
     */
    protected function getNameParts($fullName)
    {
        $parts = array_values(array_filter(explode(' ', $fullName)));
        $size = count($parts);
        $result = [];

        if (empty($parts)) {
            $result['first_name'] = null;
            $result['last_name'] = null;
        }

        if (! empty($parts) && $size == 1) {
            $result['first_name'] = $parts[0];
            $result['last_name'] = null;
        }

        if (! empty($parts) && $size >= 2) {
            $result['first_name'] = $parts[0];
            $result['last_name'] = $parts[1];
        }

        return $result;
    }

    public function acCode()
    {
        $user = auth()->user();
        $model = CustomerLinkStaff::where('customer_id',$user->id)->orderBy('id','desc')->first();

        if(!empty($model)){
            $staff_code = $model->customerStaff->ac_code;

            $char = str_replace($user->ac_code, '', $staff_code);
            $lastChar = substr($char, -1);
            if(chr(ord($lastChar)+1) == '{')
                $code = $char.'a';
            else
                $code = substr($char, 0, -1).chr(ord($lastChar)+1);
        }else{
            $code = 'a';
        }

        $code = $user->ac_code.''.$code;

        return $code;
    }

    /**
     * @param User $user
     *
     * @throws GeneralException
     * @throws \Exception
     * @throws \Throwable
     * @return User
     */
    public function forceDelete(User $user) : User
    {
        // if ($user->deleted_at === null) {
        //     throw new GeneralException(__('exceptions.backend.access.users.delete_first'));
        // }

        return DB::transaction(function () use ($user) {
            // Delete associated relationships
            $user->passwordHistories()->delete();
            $user->providers()->delete();

            if ($user->forceDelete()) {
                event(new UserPermanentlyDeleted($user));

                return $user;
            }

            throw new GeneralException(__('exceptions.backend.access.users.delete_error'));
        });
    }

}