<?php

namespace App\Services;

use App\Models\Backend\PhoneLog;
use App\Models\Auth\User;
use App\Models\Backend\Notification;
use Illuminate\Http\Request;
use App\Custom\DropboxComponent;
use App\Models\Backend\Order;
use App\Models\Backend\StockOrder;
use App\Models\Backend\RxOrders;
use DB;

class CommentService
{
    public function handleRequest(Request $request, $isAdmin = false)
    {
        if (!$request->ajax()) {
            return response()->json(['status' => false]);
        }

        try {
            switch ($request->get('type')) {
                case 'invoice':
                    return $this->handleInvoiceRequest($request);
                case 'comments':
                    return $this->handleCommentsRequest($request, $isAdmin);
                case 'add_comment':
                    return $this->handleAddCommentRequest($request, $isAdmin);
                case 'update_comment':
                    return $this->handleUpdateCommentRequest($request, $isAdmin);
                case 'delete_comment':
                    return $this->handleDeleteCommentRequest($request, $isAdmin);
                default:
                    return response()->json(['status' => false]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ]);
        }
    }

    protected function handleInvoiceRequest(Request $request)
    {
        $file = DropboxComponent::getInvoice($request->get('id'));
        if (!empty($file)) {
            return response()->json([
                'status' => true,
                'metaData' => $file
            ]);
        }
        return response()->json(['status' => false]);
    }

    protected function handleCommentsRequest(Request $request, $isAdmin)
    {
        $id = $request->get('id') ?: $request->get('comments');
        $order = Order::where('id', $id)->first();
        if (!$order) {
            return response()->json(['status' => false, 'message' => 'Order not found']);
        }

        $logs = PhoneLog::with(['replies' => function($query) {
            $query->with('user')->orderBy('created_at', 'asc');
        }])
            ->where('order_id', $id)
            ->whereNull('replied')
            ->when(!$isAdmin, function($query) {
                $query->where('type', '!=', 'notes');
            })
            ->orderBy('created_at', 'desc')
            ->get();

        $user = auth()->user();

        if(!$isAdmin) {
            $notificationUserId = $user->getCustomerAdmin()->id ? $user->getCustomerAdmin()->id : $user->getSuperAdminId();
            Notification::readAllByUser(
                $notificationUserId,
                $id,
                null,
                $type='comments',
                $isAdmin
            );
        }

        $orderType = $order->type;
        
        $rxOrder =  $rxOrderArray =  $stockOrder = null;
        
        // Only get RxOrder data if order type is 'rx'
        if ($orderType === 'rx') {
            $rxOrder = RxOrders::where('order_id', $order->id)->first();
            $rxOrderArray = $rxOrder ? $rxOrder->getShowData() : null;
        }
        
        // Get stock order if needed (assuming this might be used for other order types)
        $stockOrder = StockOrder::where('order_id', $id)->first();

        return response()->json([
            'status' => true,
            'metaData' => view('partials.chatbot_grid', [
                'logs' => $logs,
                'user' => $user,
                'sender' => $isAdmin ? 'admin': 'customer',
                'showButtons' => false,
                'isCustomerAdmin' => $user->isCustomerAdmin(),
                'type' => $orderType,
                'order' => $order,
                'stockOrder' => $stockOrder,
                'rxOrder' => $rxOrder, // will be null if not rx order
                'rxOrderArray' => $rxOrderArray, // will be null if not rx order
            ])->render(),
            'id' => $id
        ]);
    }

    protected function handleAddCommentRequest(Request $request, $isAdmin)
    {
        $reply = false;
        $updatedCount =0;
        $data = [
            'user_id' => $isAdmin ? auth()->user()->getCustomerAdmin()->id : auth()->user()->id,
            'order_id' => $request->get('id'),
            'type' => $request->get('atype'),
            'description' => $request->get('description')
        ];


         // Check if conditions for decreasing notification counter are met
        $shouldDecreaseCounter = !empty($request->get('cmtId')) && 
        $request->get('type') == 'add_comment' && 
        $request->get('atype') == 'reply';


        if (!empty($request->get('cmtId'))) {
            $data['replied'] = $request->get('cmtId');
            $data['type'] = PhoneLog::TYPE_REPLY;
            $reply = true;
        }
        $log = PhoneLog::store($data);
        if ($log) {
            $user = auth()->user();
            $isQuickReply = in_array($request->get('description'), ['Yes', 'No', 'Ok', 'Done']);

            if ($reply || $isQuickReply) {

                $notificationUserId = $isAdmin ? $user->getSuperAdminId() : $user->getCustomerAdmin()->id;
                $updatedCount = Notification::readAllByUser($notificationUserId, $request->get('id'), $request->get('cmtId'), $type='add_comment', $isAdmin);
                $parentLog = PhoneLog::where('replied', $data['replied'])->latest('id')->first();

                if ($shouldDecreaseCounter) {
                    $decreasedCounter = true;
                }
                $metaData = view('partials.chatbot_reply', [
                    'request' => $request->get('atype'),
                    'log' => $log,
                    'user' => $user,
                    'sender' => $isAdmin ? 'admin': 'customer',
                    'isCurrentUser' => $log->user_id == auth()->id(),
                    'isAdmin' => auth()->user()->isAdmin(),
                    'ownerClass' => $log->user_id == auth()->id() ? 'current-user' : ''
                ])->render();
            } else {
                $metaData = view('partials.chatbot', [
                    'log' => $log,
                    'user' => $user,
                    'sender' => $isAdmin ? 'admin': 'customer',
                    'showButtons' => $log->user_id == auth()->id()
                ])->render();
            }

            if (!$log->isNote()) {
                if (!$isAdmin) {
                    $admin = User::getUserByRole(config('access.users.admin_role'));
                    $message = "You've received message from ".$user->name.' in Order#'.$log->order->getOrderNumber();
                    Notification::addNew($message, $user->id, $admin->id, $log->order->id, $log->id, $reply);
                    $admin->sendWebNotifications([
                        'title' => $message,
                        'body' => $user->fullName().' replied in Order#'.$log->order->getOrderNumber(),
                        'data' => [
                            'id' => $parentLog->id ?? $log->id,
                            'to' => $admin->id,
                            'order_id' => $log->order->id,
                            'type' => 'comment',
                            'is_notify' => true,
                            'html' => $metaData,
                            'is_reply' => $reply || $isQuickReply,
                            'parent_id' => $log->replied ?? null,
                            'sender' => $isAdmin ? 'customer': 'admin',
                            'should_decrease_counter' => $shouldDecreaseCounter // Add this flag
                        ]
                    ]);
                } else {
                    $patientName = $log->order->patient ?? 'Unknown Patient';
                    $message = $log->getTypeText()." from Lens Shapers for Order ".$log->order->getOrderNumber()." (".$patientName.")";
                    Notification::addNew($message, $user->getSuperAdminId(), $log->order->user_id, $log->order->id, $log->id, $reply);
                    $log->order->user->sendWebNotifications([
                        'title' => $message,
                        'body' => 'Lens Shapers added '.$log->getTypeText().' in Order#'.$log->order->getOrderNumber(),
                        'data' => [
                            'id' => $parentLog->id ?? $log->id,
                            'to' => $log->order->user_id,
                            'order_id' => $log->order->id,
                            'type' => 'comment',
                            'is_notify' => true,
                            'html' => $metaData,
                            'is_reply' => $reply || $isQuickReply,
                            'parent_id' => $log->replied ?? null,
                            'sender' =>  $isAdmin ? 'admin': 'customer',
                            'should_decrease_counter' => $shouldDecreaseCounter // Add this flag

                        ]
                    ]);
                }
            }
            return response()->json([
                'status' => true,
                'metaData' => $metaData,
                'id' => $parentLog->id ?? $log->id,
                'is_reply' => $reply || $isQuickReply,
                'replied' => $log->replied,
                'sender_id' => auth()->id(),
                'updatedCount' => $updatedCount,
                // 'should_decrease_counter' => $shouldDecreaseCounter,
                'should_decrease_counter' => $isAdmin && $reply,
                'sender' => $isAdmin ? 'admin' : 'customer',
                'comment' => [
                    'order_id' => $log->order_id,
                    'parent_id' => $log->replied ?? null,
                    'user_to' => $isAdmin ? ($admin->id ?? null) : ($log->order->user_id ?? null),
                    'is_reply' => $reply || $isQuickReply,
                    'replied' => $log->replied ?? null,
                    'sender' => $isAdmin ? 'admin': 'customer'
                ]
            ]);
        }
        return response()->json(['status' => false]);
    }

    protected function handleUpdateCommentRequest(Request $request, $isAdmin)
    {
        DB::beginTransaction();
        
        try {
            $log = PhoneLog::with(['order.user', 'user'])->findOrFail($request->get('id'));
            $log->description = $request->get('description');
            $log->save();

            $user = auth()->user();
            $isReply = !empty($log->replied);
            // Render HTML for sender (with buttons) and receiver (without buttons)
            $senderHtml = view('partials.chatbot', [
                'log' => $log,
                'user' => auth()->user(),
                'showButtons' => true // Show buttons to sender
            ])->render();

            $receiverHtml = $isReply 
            ? view('partials.chatbot_reply', [
                'log' => $log,
                'user' => $user,
                'sender' => $isAdmin ? 'admin' : 'customer',
                'isCurrentUser' => $log->user_id == auth()->id(),
                'isAdmin' => auth()->user()->isAdmin(),
                'ownerClass' => $log->user_id == auth()->id() ? 'current-user' : ''
              ])->render()
            : view('partials.chatbot', [
                'log' => $log,
                'user' => $user,
                'showButtons' => false
              ])->render();

            // Prepare notification data
            $notificationData = [
                'id' => $log->id,
                'order_id' => $log->order->id,
                'type' => 'update_comment',
                'is_notify' => false,
                'is_reply' => $isReply,
                'replied' => $log->replied,
                'sender' => $isAdmin ? 'admin' : 'customer'
            ];


            // Send appropriate HTML based on who is receiving the notification
            $admin = User::getUserByRole(config('access.users.admin_role'));
            $customer = $log->order->user;

             if (!$isAdmin) {
                // Customer updated - notify admin
                $admin = User::getUserByRole(config('access.users.admin_role'));
                $notificationData['html'] = $receiverHtml;
                $notificationData['to'] = $admin->id;
                $notificationData['user_to'] = $admin->id;
                
                $admin->sendWebNotifications([
                    'title' => 'Comment Updated',
                    'body' => 'A comment was updated in Order#'.$log->order->getOrderNumber(),
                    'data' => $notificationData
                ]);
            } else {
                // Admin updated - notify customer
                $notificationData['html'] = $receiverHtml;
                $notificationData['to'] = $log->order->user_id;
                $notificationData['user_to'] = $log->order->user_id;
                if($log->type !=  'notes') {
                    $log->order->user->sendWebNotifications([
                        'title' => 'Comment Updated',
                        'body' => 'Your comment was updated in Order#'.$log->order->getOrderNumber(),
                        'data' => $notificationData
                    ]);
                }
            }
            DB::commit();

            return response()->json([
                'status' => true,
                'html' => $senderHtml,
                'metaData' => $senderHtml,
                'comment' => [
                    'id' => $log->id,
                    'order_id' => $log->order_id,
                    'type' => 'update_comment',
                    'html' => $senderHtml,
                    'to' => $isAdmin ? $log->order->user_id : $admin->id,
                    'user_to' => $isAdmin ? $log->order->user_id : $admin->id,
                    'replied' => $log->replied,
                    'is_reply' => $isReply,
                    'is_notify' => true,
                    'sender' => $isAdmin ? 'admin' : 'customer'
                ]
            ]);


        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ]);
        }
    }

    protected function handleDeleteCommentRequest(Request $request, $isAdmin)
    {
        DB::beginTransaction();
        $updatedCount = 0;   
        try {
            $log = PhoneLog::with(['order.user'])->findOrFail($request->get('id'));
            
            $orderId = $log->order_id;
            $logId = $log->id;
            $userId = $log->user_id;
            $isReply = !empty($log->replied);

            // Delete the comment
            $log->delete();
            // Delete related notifications
            $deletedNotifications = Notification::where('plog_id', $logId)->delete();

            $user = auth()->user();
            // Prepare notification data
            $notificationData = [
                'id' => $logId,
                'order_id' => $orderId,
                'type' => 'delete_comment',
                'is_notify' => false,
                'user_id' => $userId,
                'deleted_by_admin' => $isAdmin,
                'is_reply' => $isReply,
                'replied' => $log->replied ?? null
            ];

             // Always update notification counter for both threads and replies
            $notificationUserId = $isAdmin ? $user->getSuperAdminId() : $user->getCustomerAdmin()->id;
            $updatedCount = Notification::readAllByUser(
                $notificationUserId, 
                $orderId, 
                $logId, 
                $isReply ? 'replies' : 'comments',
                $isAdmin
            );


            // Send notification to the other side
            if (!$isAdmin) {
                // Customer deleted - notify admin
                $admin = User::getUserByRole(config('access.users.admin_role'));
                $admin->sendWebNotifications([
                    'title' => 'Comment Deleted',
                    'body' => 'A comment was deleted in Order#'.$log->order->getOrderNumber(),
                    'data' => array_merge($notificationData, [
                        'to' => $admin->id,
                        'user_to' => $admin->id,
                        'sender' => $isAdmin ? 'admin': 'customer',
                        'updatedCount' => $updatedCount,
                        'should_decrease_counter' => true 
                    ])
                ]);
            } else {
                 // Admin deleted - notify customer
                 $log->order->user->sendWebNotifications([
                    'title' => 'Comment Deleted',
                    'body' => 'A comment was deleted in Order#'.$log->order->getOrderNumber(),
                    'data' => array_merge($notificationData, [
                        'to' => $log->order->user_id,
                        'user_to' => $log->order->user_id,
                        'sender' => $isAdmin ? 'admin': 'customer',
                        'updatedCount' => $updatedCount,
                        'should_decrease_counter' => true
                    ])
                ]);
            }
            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Comment deleted successfully',
                'comment' => [ // Include this to match other operations
                    'id' => $logId,
                    'order_id' => $orderId,
                    'type' => 'delete_comment',
                    'sender' => $isAdmin ? 'admin' : 'customer',
                    'is_reply' => $isReply,
                    'replied' => $log->replied ?? null,
                    'updatedCount' => $updatedCount,
                    'should_decrease_counter' => true
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ]);
        }
    }
}