(function() {
    function initChatbot() {
        // Initialize with current user ID
        const isSuperAdminDefined = typeof isSuperAdmin !== 'undefined' ? isSuperAdmin : false;
        const currentUserId = isSuperAdminDefined ? '1' : (typeof userId !== 'undefined' ? userId : '1');
        document.body.dataset.currentUser = currentUserId;

        // Setup delete button timeouts
        setupDeleteButtonTimeouts();
        setupCommentBox();
        // Request notification permission
        if ('Notification' in window && Notification.permission !== "granted") {
            Notification.requestPermission();
        }


        function toastMessage(level,text){
            if(text != '' && level != ''){
                switch(level){
                    case 'success':
                        toastr.success(text)
                    break;
                    case 'error':
                        toastr.error(text)
                    break;
                    case 'info':
                        toastr.info(text)
                    break;
                    case 'warning':
                        toastr.warning(text)
                    break;
                }
            }
        }

        // Pusher notification channel handlers
        if (typeof notificationChannel !== 'undefined' && notificationChannel) {
            notificationChannel.bind('send', function(payload) {
            const { data } = payload;
            const notificationData = {
                ...data.data,
                sender_id: data.user_id,
                order_id: data.data.order_id,
                id: data.data.id,
                parent_id: data.data.parent_id || data.data.replied || null,
                is_reply: data.data.type === 'reply',
                html: data.data.html,
                is_notify: data.data.is_notify,
                user_to: data.data.user_to,
                replied: data.data.replied,
                type: data.data.type,
                to: data.data.to,
                should_decrease_counter: data.data.should_decrease_counter || false,
                sender: data.data.sender || 'customer',
                // updatedCount: data.data.updatedCount || 0
                updatedCount: 1
            };
            // Handle delete comment
            if (notificationData.type == 'delete_comment') {
                removeCommentFromUI(notificationData.id);
                if (notificationData.updatedCount) {
                    updateNotificationCounters(notificationData.order_id, -notificationData.updatedCount);
                }
                return;
            }
            // Handle notification counters
            // handleNotificationCounters(notificationData);

            // Show notification if applicable
            if (shouldShowNotification(notificationData)) {
                showNotification(data.title, notificationData.order_id);
            }

            // Handle message types
            if (['comment', 'reply', 'update_comment'].includes(notificationData.type)) {
                processIncomingMessage(notificationData);
            }
        });

        }
         // Helper functions
        function shouldShowNotification(notificationData) {
            return notificationData.is_notify && 
                (notificationData.to == currentUserId || 
                 (notificationData.type == 'reply' && notificationData.replied == currentUserId) || 
                 notificationData.type == 'comment' ||
                 notificationData.type == 'update_comment');
        }

        function handleNotificationCounters(notificationData) {
            // Log initial notification data for debugging
            console.group('Notifciation handler handleNotificationCounters()')
            console.log('handleNotificationCounters called with:', notificationData);

            const shouldUpdate = (
                notificationData.should_decrease_counter || 
                (notificationData.is_reply && notificationData.sender === 'admin')
            );
            
            if (!shouldUpdate) {
                console.log('No update needed: shouldUpdate is false');
                return;
            }

            // Get the current order ID
            const currentOrderId = getCurrentOrderId();
            console.log('currentOrderId :', currentOrderId)

            const orderId = notificationData.comment.order_id;
            const isCurrentOrder = currentOrderId  == orderId; // Ensure type consistency

            console.log('isCurrentOrder :', isCurrentOrder)
            console.log('Current Order ID:', currentOrderId, 'Notification Order ID:', orderId, 'Is Current Order:', isCurrentOrder);

            // Update general notification counters (header)
            $('.notifications-badge-counter').each(function() {
                const $counter = $(this);
                let currentCount = parseInt($counter.text()) || 0;
                const decrementBy = Math.max(0, notificationData.updatedCount || 1);
                let newCount = Math.max(0, currentCount - decrementBy);
                
                console.log('Updating .notifications-badge-counter:', { currentCount, decrementBy, newCount });
                $counter.text(newCount);
                $counter.toggle(newCount > 0);
            });

            // Update tab counter if this is the current order
            if (isCurrentOrder) {
                const $tabCounters = $('.noti_counter-span');
                if ($tabCounters.length == 0) {
                    console.warn('No .noti_counter-span elements found in DOM');
                }
                $tabCounters.each(function() {
                    const $counter = $(this);
                    let currentCount = parseInt($counter.text()) || 0;
                    const decrementBy = Math.max(0, notificationData.updatedCount || 1);
                    let newCount = Math.max(0, currentCount - decrementBy);
                    
                    console.log('Updating .noti_counter-span:', { currentCount, decrementBy, newCount });
                    $counter.text(newCount);
                    $counter.toggle(newCount > 0);
                });
            } else {
                console.log('Skipping .noti_counter-span update: not current order');
            }

            // Update specific order's notification counter
            const $orderCounter = $('.split-table #tr_order-' + orderId).find('.order_notification-counter');
            if ($orderCounter.length) {
                let current = parseInt($orderCounter.text()) || 0;
                const decrementBy = Math.max(0, notificationData.updatedCount || 1);
                let newCount = Math.max(0, current - decrementBy);
                
                console.log('Updating .order_notification-counter:', { current, decrementBy, newCount });
                $orderCounter.text(newCount);
                $orderCounter.toggle(newCount > 0);
            } else {
                console.warn('No .order_notification-counter found for order ID:', orderId);
            }
            console.groupEnd();
        }

        function processIncomingMessage(messageData) {
            const currentOrderId = getCurrentOrderId();
            if (currentOrderId != messageData.order_id) return;

            // Process HTML to remove edit/delete buttons if needed
            const processedMessageData = processMessageHTML(messageData);

            if (messageData.type == 'update_comment') {
                handleCommentUpdate(processedMessageData);
            } else {
                appendNewMessage(processedMessageData);
            }
        }

        function processMessageHTML(messageData) {
            const processed = {...messageData};
            if (processed.to == currentUserId) {
                try {
                    const tempDiv = document.createElement('div');
                    tempDiv.innerHTML = processed.html;
                    // Remove delete buttons
                    tempDiv.querySelectorAll('.comp-cmt-tab-delete-btn').forEach(btn => {
                        btn.parentNode.removeChild(btn);
                    });
                    // Remove edit buttons
                    tempDiv.querySelectorAll('.comp-cmt-tab-edit-btn').forEach(btn => {
                        btn.parentNode.removeChild(btn);
                    });
                    processed.html = tempDiv.innerHTML;
                } catch (e) {
                    console.error('Error processing message HTML:', e);
                    processed.html = messageData.html;
                }
            }
            return processed;
        }

        function handleCommentUpdate(messageData) {
            const $existingComment = $(`.comment-${messageData.id}`);
            if ($existingComment.length) {
                // Only update the content parts that changed
                const $commentBody = $existingComment.find('.media-body');
                // Update the comment content
                $commentBody.html(messageData.html);
                // Re-apply styles and attributes
                applyMessageStyles($existingComment, messageData);
                // Reinitialize any interactive elements
                setupDeleteButtonTimeouts();
                // Maintain the fade animation for visual consistency
                $existingComment.hide().fadeIn(300);
            } else {
                // Fallback to append if comment doesn't exist
                appendNewMessage(messageData);
            }
        }

        function appendNewMessage(messageData) {
            const isReply = messageData.parent_id || messageData.is_reply;
            const parentId = messageData.parent_id || messageData.replied;
            const sender = messageData.sender || 'customer';
            
            if (isReply) {
                appendReplyMessage(messageData, parentId, sender);
            } else {
                addAsNewThread(messageData);
            }
            // Apply styles to newly added messages
            const $newMessage = $(`.comment-${messageData.id}`);
            applyMessageStyles($newMessage, messageData);
            setupDeleteButtonTimeouts();
        }


        function applyMessageStyles($message, messageData) {
            if (!$message.length) return;
            // Find the comment content element
            const $commentContent = $message.find('.comment-content');
            if (!$commentContent.length) return;
            // Clear existing classes
            $commentContent.removeClass('current-user-bg other-user-bg');
            // Apply appropriate background class based on both conditions
            if (messageData.is_reply == true) {
                $commentContent.addClass('current-user-bg');
            } else {
                $commentContent.addClass('other-user-bg');
            }
                // Still set data attributes for consistency
            $message.attr('data-owner', messageData.sender_id == currentUserId ? 'current-user' : 'other-user');
            
            if (messageData.is_reply) {
                $message.addClass('reply');
            }
        }

        function appendReplyMessage(messageData, parentId, sender) {
            const $parent = $(`.comment-list .comment-${parentId}, .comment-${parentId}`);
            if ($parent.length) {
                let $repliesContainer = $parent.find('.replies-container');
                if (!$repliesContainer.length) {
                    $repliesContainer = $('<div class="replies-container"></div>');
                    $parent.find('.media-body').append($repliesContainer);
                }
                const replyHtml = `
                    <div class="media comment comment-${messageData.id} reply" 
                        data-id="${messageData.id}" 
                        data-sender="${sender}"
                        data-owner="${messageData.sender_id == currentUserId ? 'current-user' : 'other-user'}">
                        <div class="media-body todo-comment">
                            ${messageData.html}
                        </div>
                    </div>
                `;
            const $newReply = $(replyHtml).hide();
            $repliesContainer.append($newReply);
            $newReply.fadeIn(300, function() {
                // Scroll to bottom to show the recent message
                setTimeout(() => {
                    const $commentsContainer = $('.comment-box-container');
                    $commentsContainer.scrollTop($commentsContainer[0].scrollHeight);
                    if ($commentsContainer[0].Ps) {
                        $commentsContainer[0].Ps.update();
                    }
                }, 10);
            });
            } else {
                addAsNewThread(messageData);
            }
        }

        function addAsNewThread(messageData) {
            const $messageList = $('.comment-list, .media-list.comment-list');
            if ($messageList.length && !$messageList.find(`.comment-${messageData.id}`).length) {
                const $newMessage = $(messageData.html).hide();
                $messageList.prepend($newMessage);
                $newMessage.fadeIn(300, function() {
                    // Scroll to bottom of comment list to show recent message
                    setTimeout(() => {
                        const $commentsContainer = $('.comment-box-container');
                        $commentsContainer.scrollTop($commentsContainer[0].scrollHeight);
                        if ($commentsContainer[0].Ps) {
                            $commentsContainer[0].Ps.update();
                        }
                    }, 10);
                });
                $messageList.find('#cmnt-remove-empty-txt').remove();
            }
        }

        // Helper functions
        function getCurrentOrderId() {
            const orderIdInput = $('#cmt_order_id');
            if (orderIdInput.length) {
                const orderId = orderIdInput.val();
                return parseInt(orderId) || null;
            }
        
            const activeRow = $('tr.active[data-id]');
            if (activeRow.length) {
                const orderId = activeRow.attr('data-id');
                return parseInt(orderId) || null;
            }
            
            const orderContainer = $('.order-details-container[data-order-id]');
            if (orderContainer.length) {
                const orderId = orderContainer.attr('data-order-id');
                return parseInt(orderId) || null;
            }
            
            const urlMatch = window.location.pathname.match(/orders\/(\d+)/);
            if (urlMatch) {
                return parseInt(urlMatch[1]) || null;
            }
            
            return null;
        }

         // Function to show notifications
           function showNotification(title, orderId) {
             toastMessage('success', title);
             webPushNotification(title);
               updateNotificationCounters(orderId);
           }

        function webPushNotification(title,body) {
            if(window.Notification) {
                Notification.requestPermission(function(status) {
                    var options = {
                        body: body,
                        // dir: 'ltr',
                        icon:$('#favicon').attr('href')
                    }
                    var n = new Notification(title, options);
                });
            }
            else {
                alert('Your browser doesn\'t support notifications.');
            }
        }
            
        function updateNotificationCounters(orderId, increment = 1) {
            // Update general notification counters
            const currentOrderId = getCurrentOrderId();
            if(currentOrderId) {
                $('.noti_counter-span').each(function() {
                    const $counter = $(this);
                    const current = parseInt($counter.text()) || 0;
                    const newCount = Math.max(0, current + increment);
                    $counter.text(newCount).toggle(newCount > 0);
                    console.groupEnd();
                });
            }
                $('.notifications-badge-counter').each(function() {
                    const $counter = $(this);
                    const current = parseInt($counter.text()) || 0;
                    const newCount = Math.max(0, current + increment);
                    $counter.text(newCount).toggle(newCount > 0);
                });
            
            // Update specific order's notification counter if it exists
            const $orderCounter = $('.split-table #tr_order-' + orderId).find('.order_notification-counter');
            if ($orderCounter.length) {
                const current = parseInt($orderCounter.text()) || 0;
                const newCount = Math.max(0, current + increment);
                $orderCounter.text(newCount).toggle(newCount > 0);
            }
        }

        function addCommentByCustomer(orderId, cmtId, message, element) {
            console.log('Quick reply:', {orderId, cmtId, message});
            
            if (!message) {
                if (element) element.find('.reply-textarea').addClass('error');
                return;
            }
        
            $.ajax({
                url: baseUrl + '/orders/details-type-api',
                type: "POST",
                dataType: 'json',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                },
                data: {
                    id: orderId,
                    cmtId: cmtId,
                    description: message,
                    type: 'add_comment',
                    _token: $('meta[name="csrf-token"]').attr('content'),
                    atype: 'reply' // Explicitly set as reply
                },
                success: function(response) {
                    console.log('Quick reply response:', response);
                    if (response.status) {
                        // Reset button state
                        $('.cmnt-reply-yes, .cmnt-reply-no, .cmnt-reply-ok, .cmnt-reply-done')
                            .prop('disabled', false)
                            .html(function() {
                                return $(this).data('original-html') || $(this).html();
                            });
                        if (response.comment) {
                            appendNewMessage({
                                comment_id: cmtId,
                                order_id: orderId,
                                parent_id: response.comment.parent_id || null,
                                html: response.metaData,
                                is_notify: true,
                                user_to: response.comment.user_to,
                                replied: response.comment.replied,
                                message_id: response.id,
                                is_reply: response.is_reply
                            });
                        }
                        handleNotificationCounters(response)
                        dispatchNotificationEvent('reply', {
                            id: orderId,
                            commentId: response.comment?.id
                        });
                    }
                },
                error: function(xhr, status, error) {
                    console.log('Quick reply error:', error);
                    // Reset button state
                    $('.cmnt-reply-yes, .cmnt-reply-no, .cmnt-reply-ok, .cmnt-reply-done')
                        .prop('disabled', false)
                        .html(function() {
                            return $(this).data('original-html') || $(this).html();
                        });
                    
                    toastMessage('error', 'Failed to send quick reply');
                }
            });
        }

        function updateComment(commentId, newText, $comment, original) {
            const $commentContent = $comment.find('.comment-content');
            
            $.ajax({
                url: baseUrl + '/orders/details-type-api',
                type: "POST",
                dataType: 'json',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                },
                data: {
                    id: commentId,
                    description: newText,
                    type: 'update_comment',
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                beforeSend: function() {
                    // Show loading state
                    $commentContent.html('<i class="fa fa-spinner fa-spin"></i> Saving...');
                },
                success: function(response) {
                    if (response.status) {
                        $commentContent.html(`${newText}`);
                        $commentContent.attr('class', original.classes);
                        $commentContent.css('width', original.width);
                        // If buttons were added, set up their timeout
                        if (response.comment.sender == "admin") {
                            setupDeleteButtonTimeouts();
                        }
                        toastMessage('success', 'Comment updated successfully');
                    } else {
                        // Restore original content if update failed
                        $commentContent.html($comment.data('original'));
                        toastMessage('error', response.message || 'Failed to update comment');
                    }
                },
                error: function(xhr, status, error) {
                    console.log('AJAX Error:', status, error);
                    // Restore original content
                    $commentContent.html($comment.data('original'));
                    toastMessage('error', 'Failed to update comment');
                }
            });
        }

        function deleteComment(commentId, $comment) {
            $.ajax({
                url: baseUrl + '/orders/details-type-api',
                type: "POST",
                dataType: 'json',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                },
                data: {
                    id: commentId,
                    type: 'delete_comment',
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                beforeSend: function() {
                    // Show loading state
                    $comment.html('<div class="media-body todo-comment"><i class="fa fa-spinner fa-spin"></i> Deleting...</div>');
                },
                success: function(response) {
                    if (response.status) {
                        // Update notification counters
                        if (response.updatedCount) {
                            updateNotificationCounters(response.order_id, -response.updatedCount);
                        }
                        // The actual removal will be handled by the Pusher notification
                        // Just show success message
                        toastMessage('success', 'Comment deleted successfully');
                        $comment.fadeOut(300, function() {
                            $(this).remove(); // Clean removal from DOM
                        });
                        
                    } else {
                        // Restore the comment
                        $comment.find('.media-body').html('Failed to delete comment');
                        // setTimeout(() => {
                        //     location.reload(); // Fallback to reload if delete fails
                        // }, 1500);
                        toastMessage('error', response.message || 'Failed to delete comment');
                    }
                },
                error: function(xhr, status, error) {
                    console.log('AJAX Error:', status, error);
                    $comment.find('.media-body').html('Error deleting comment');
                    // setTimeout(() => {
                    //     location.reload(); // Fallback to reload on error
                    // }, 1500);
                    toastMessage('error', 'Failed to delete comment');
                }
            });
        }

       // Universal comment removal function
        function removeCommentFromUI(commentId) {
            console.group('Removing comment:', commentId);
            console.log('Finding elements with selector:', `.comment-${commentId}`);
            // Find all instances of the comment (main and replies)
            const $comments = $(`.comment-${commentId}`);
            console.log('Found', $comments.length, 'matching elements');
            if ($comments.length == 0) {
                console.warn('No comments found with ID:', commentId);
                return;
            }
            
            $comments.each(function(index) {
                console.log('Processing element', index + 1);
                const $comment = $(this);

                const $parent = $comment.parent();
                console.log('Parent element:', $parent);
                console.log('Parent class:', $parent.attr('class'));
                
                // First fade out and remove any replies to this comment
                $comment.find('.replies-container .comment').fadeOut(300, function() {
                    $(this).remove();
                });
                
                // Then fade out and remove the comment itself
                $comment.fadeOut(300, function() {
                    $(this).remove();
                    
                    // Clean up empty reply containers
                    const $parentComment = $comment.closest('.media.comment');
                    if ($parentComment.length) {
                        const $repliesContainer = $parentComment.find('.replies-container');
                        if ($repliesContainer.length && $repliesContainer.children().length == 0) {
                            $repliesContainer.remove();
                        }
                    }
                    checkEmptyComments();
                });
            });
            console.groupEnd();
        }

        function checkEmptyComments() {
            // Show "no comments" message if applicable
            if ($('.comment-list .comment').length === 0) {
                $('.comment-list').append('<p id="cmnt-remove-empty-txt" class="text-center">No Comments for this order</p>');
            }
        }

        function dispatchNotificationEvent(action, notificationData) {
            const currentUserId = typeof userId !== 'undefined' ? userId : 1;
            fetch('/dashboard/recent-notification', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    orderId: notificationData.id,
                    action: action,
                    portalType: portalType,
                    // userId: currentUserId
                    userId: portalType == 'admin' ? 1 : currentUserId
                })
            })
            .then(response => response.json())
            .then(data => {
                const event = new CustomEvent(`comment_${action}`, {
                    detail: {
                        notification: data.response
                    }
                });
                window.dispatchEvent(event);
            })
            .catch(error => {
                toastMessage('error', 'Failed to send notification');
            });
        }

        // Event handlers for comment actions
        $(document).on('click', '.cmnt-reply', function(e) {
            e.preventDefault();
            
            const $container = $(this).closest('.media-body');
            $container.find('.comment-actions').hide();
            const $replyForm = $container.find('.reply-form-container').show();
            $replyForm.find('.reply-textarea').focus();
            
            const $commentsContainer = $('.comment-box-container');
            const containerRect = $commentsContainer[0].getBoundingClientRect();
            const formRect = $replyForm[0].getBoundingClientRect();

            // Calculate the full height including buttons
            const $buttons = $replyForm.find('.reply-form-container .submit-reply, .reply-form-container .cancel-reply');
            const buttonsHeight = $buttons.length ? $buttons.outerHeight(true) : 0;

            // Check if form or buttons are below visible area
            if (formRect.bottom + buttonsHeight > containerRect.bottom) {
                setTimeout(() => {
                    const scrollTo = $replyForm.offset().top - $commentsContainer.offset().top + $commentsContainer.scrollTop() - 20;
                    $commentsContainer.animate({
                        scrollTop: scrollTo + buttonsHeight
                    }, 100);
                }, 1);
            }
        });

        $(document).on('click', '.submit-reply', function() {
            const $container = $(this).closest('.reply-form-container');
            const commentId = $(this).data('id');
            const orderId = $(this).data('oid');
            const message = $container.find('.reply-textarea').val().trim();
            
            if (message) {
                addCommentByCustomer(orderId, commentId, message, $(this).closest('.media'));
                $container.find('.reply-textarea').val('').height('100px');
                $container.hide();
                $container.siblings('.comment-actions').show();
            }
        });

        $(document).on('click', '.cancel-reply', function() {
            const $container = $(this).closest('.reply-form-container');
            $container.find('.reply-textarea').val('').height('100px');
            $container.hide();
            $container.siblings('.comment-actions').show();
        });

        
        // Initialize all quick reply buttons
        document.addEventListener('click', function(e) {
            const yesBtn = e.target.closest('.cmnt-reply-yes');
            const noBtn = e.target.closest('.cmnt-reply-no');
            const okBtn = e.target.closest('.cmnt-reply-ok');
            const doneBtn = e.target.closest('.cmnt-reply-done');
            
            if (yesBtn) {
                e.stopPropagation();
                handleQuickReplyAction(yesBtn, 'Yes');
                return;
            }
            if (noBtn) {
                e.stopPropagation();
                handleQuickReplyAction(noBtn, 'No');
                return;
            }
            if (okBtn) {
                e.stopPropagation();
                handleQuickReplyAction(okBtn, 'Ok');
                return;
            }
            if (doneBtn) {
                e.stopPropagation();
                handleQuickReplyAction(doneBtn, 'Done');
                return;
            }
        });


        // Quick reply buttons handler
        function handleQuickReplyAction(button, replyText) {
            const orderId = button.dataset.oid;
            const commentId = button.dataset.id;
              // Show loading state
            // button.disabled = true;
            // button.classList.remove('btn-secondary');
            // button.classList.add('btn-primary');
            // button.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Sending...';
            // Call addCommentByCustomer
            addCommentByCustomer(orderId, commentId, replyText, $(button).closest('.media'));
        }

        

        // Comment management
        $(document).on('click', '.comp-cmt-tab-delete-btn', function(e) {
            e.preventDefault();
            if (!confirm('Are you sure you want to delete this comment?')) return;
            const commentId = $(this).data('id');
            const $comment = $(this).closest('.comment');
            deleteComment(commentId, $comment);
        });

        $(document).on('click', '.comp-cmt-tab-edit-btn', function(e) {
            e.preventDefault();
            const $comment = $(this).closest('.comment');
            const $commentContent = $comment.find('.comment-content');
            const $actionButtons = $comment.find('.comment-actions-append, .comment-actions-append-reply');
            const currentText = $commentContent.contents().filter(function() {
                return this.nodeType === 3 && this.nodeValue.trim() !== '';
            }).first().text().trim();
            
            // Store original classes along with HTML
            $comment.data('original', {
                html: $commentContent.html(),
                classes: $commentContent.attr('class'),
                width: $commentContent.css('width'),
                actionButtonsVisible: $actionButtons.is(':visible')
            });

            // Hide action buttons
            $actionButtons.hide();

            // Set width to 94% with !important when editing
            $commentContent.attr('style', function(i, style) {
                return (style || '') + 'width: 94% !important;';
            });
            
            // Remove width-restricting classes
            $commentContent.removeClass('too-short-text short-text').css({
                'max-width': '',
                'min-width': ''
            });
            
            $commentContent.html(`
                <div class="edit-comment-wrapper">
                    <textarea class="form-control edit-comment-textarea">${currentText}</textarea>
                    <div class="edit-actions">
                        <button class="comp-cmt-tab-cancel-btn btn btn-outline-secondary btn-sm me-2 red">
                            <i class="fa fa-times"></i> Cancel
                        </button>
                        <button class="comp-cmt-tab-save-btn btn btn-primary btn-sm blue" data-id="${$(this).data('id')}">
                            <i class="fa fa-check"></i> Save
                        </button>
                    </div>
                </div>
            `);
        });
        
        $(document).on('click', '.comp-cmt-tab-cancel-btn', function(e) {
            e.preventDefault();
            const $comment = $(this).closest('.comment');
            const $commentContent = $comment.find('.comment-content');
            const originalData = $comment.data('original');
            
            // Restore original HTML and classes
            $commentContent.html(originalData.html)
                        .attr('class', originalData.classes)
                        .css('width', originalData.width); // Revert to original width

        });
        
        $(document).on('click', '.comp-cmt-tab-save-btn', function(e) {
            e.preventDefault();
            const commentId = $(this).data('id');
            const $comment = $(this).closest('.comment');
            const original = $comment.data('original');
            const newText = $comment.find('.edit-comment-textarea').val().trim();
            if (newText) updateComment(commentId, newText, $comment, original);
        });


        // Textarea auto-resize
        $(document).on('input', '.reply-textarea, .edit-comment-textarea', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });

        // Comment box functionality
        let selectedType = '';
        const commentBox = document.getElementById('comment-box');
        const commentTextarea = document.getElementById('comment_textarea');
        const textareaButtons = document.getElementById('textarea-buttons');
        const actionButtons = document.getElementById('action-buttons');
        const cancelBtn = document.getElementById('cancel-comment-btn');
        const sendBtn = document.getElementById('send-comment-btn');
        const textareaHeader =  document.getElementById('textarea-header');
        

        if (cancelBtn && sendBtn && commentTextarea) { 
            cancelBtn.addEventListener('click', function() {
                commentTextarea.value = '';
                commentTextarea.style.display = 'none';
                textareaHeader.style.display = 'none';
                textareaButtons.style.display = 'none';
                actionButtons.style.display = 'block';

            });

            function handleActionButtonClick(btn) {
                selectedType = btn.getAttribute('data-type');
                const placeholder = btn.getAttribute('data-placeholder');
                const type = btn.getAttribute('data-type');
                const id = document.getElementById('cmt_order_id').value;
                
                if (type === '<?= PhoneLog::TYPE_ORDER_CANCELLATION ?>') {
                    Swal.fire({
                        title: "Order cancellation is not guaranteed. We will try to cancel the order if possible and confirm to you.\n\nAre you sure you want to cancel this order?",
                        icon: "warning",
                        showCancelButton: true,
                        confirmButtonColor: "#DD6B55",
                        confirmButtonText: "Yes, request cancellation",
                        cancelButtonText: "No",
                    }).then((result) => {
                        if (result.isConfirmed) {
                            showCommentTextarea(placeholder);
                        }
                    });
                    return;
                }
                
                showCommentTextarea(placeholder);
            }

            function showCommentTextarea(placeholder) {
                commentTextarea.placeholder = placeholder;
                commentTextarea.style.display = 'block';
                commentTextarea.focus();
                textareaButtons.style.display = 'block';
                actionButtons.style.display = 'none';
                commentTextarea.style.height = '100px';

            }

            // Event delegation for all action buttons
            const buttonsContainer = document.getElementById('action-buttons') || document.body;
            buttonsContainer.addEventListener('click', function(e) {
                const btn = e.target.closest('[data-type]');
                if (btn) {
                    handleActionButtonClick(btn);
                }
            });

            sendBtn.addEventListener('click', function() {
                const comment = commentTextarea.value.trim();
                const type = selectedType;
                const id = document.getElementById('cmt_order_id').value;
                const textareaHeader =  document.getElementById('textarea-header');
                if (comment) {
                    addCommentApi(id, comment, type);
                    commentTextarea.value = '';
                    commentTextarea.style.display = 'none';
                    textareaButtons.style.display = 'none';
                    textareaHeader.style.display = 'none';
                    actionButtons.style.display = 'block';

                } else {
                    commentTextarea.classList.add('error');
                    commentTextarea.focus();
                }
            });

            commentTextarea.addEventListener('input', function() {
                this.style.height = 'auto';
                this.style.height = (this.scrollHeight) + 'px';
            });
        }

        function addCommentApi(id, message, aType) {
        console.log('-------addCommentApi---------------------')

        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        const data = {
            id: id,
            type: 'add_comment',
            description: message,
            _token: csrfToken,
            atype: aType,
            user_to: null
        };
        
        fetch(baseUrl + '/orders/details-type-api', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify(data)
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            console.log(data, '-------------------data---------------')
            if (data.status) {
                document.getElementById('comment_textarea').value = '';
                const commentList = document.querySelector('.comment-list');
                if (commentList) {
                    const newComment = document.createElement('div');
                    newComment.innerHTML = data.metaData;
                    newComment.style.display = 'none'; // Hide initially for fade-in effect
                    commentList.appendChild(newComment);
                    $(newComment).fadeIn(300, function() {
                        // Scroll to bottom of comment list to show recent message
                        setTimeout(() => {
                            const $commentsContainer = $('.comment-box-container');
                            $commentsContainer.scrollTop($commentsContainer[0].scrollHeight);
                            if ($commentsContainer[0].Ps) {
                                $commentsContainer[0].Ps.update();
                            }
                        }, 10);
                    });
                }
                const emptyText = document.getElementById('cmnt-remove-empty-txt');
                if (emptyText) {
                    emptyText.style.display = 'none';
                }
                if (data.comment) {
                    appendNewMessage({
                        order_id: data.comment.order_id,
                        parent_id: data.comment.parent_id,
                        html: data.comment.html,
                        is_notify: true,
                        user_to: data.comment.user_to
                    });
                }
                if( aType != 'notes') {
                    dispatchNotificationEvent('added', {
                        id: data.comment.order_id,
                    });
                }
            }
        })
        .catch(error => {
            console.log('Error:', error.message);
        });
    }

        // Function to hide delete buttons when timeout expires
        function setupDeleteButtonTimeouts() {
            $('.comp-cmt-tab-delete-btn, .comp-cmt-tab-edit-btn').each(function() {
                const $button = $(this);
                const rawTime = $button.data('time');
                // Parse as local time since the timestamp is in app's timezone format
                const createdAt = new Date(Date.parse(rawTime.replace(' ', 'T')));
                // Validate createdAt
                if (isNaN(createdAt.getTime())) {
                    $button.remove();
                    return;
                }

                const now = new Date();
                const diffSeconds = Math.max(0, (now - createdAt) / 1000);
                const remainingTime = Math.max(0, (30 - diffSeconds) * 1000);
                if (remainingTime > 0) {
                    setTimeout(function() {
                        $button.fadeOut(300, function() {
                            $(this).remove();
                        });
                    }, remainingTime);
                } else {
                    $button.remove();
                }
            });
        }

        // Updated code to resize message divs according to character limit
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                mutation.addedNodes.forEach((node) => {
                    if (node.nodeType === 1) { // Element node
                        const comments = node.querySelectorAll ? node.querySelectorAll('.comment-content:not(.replies-container .comment-content)') : [];
                        comments.forEach((element) => {
                            // Reset any previous width settings
                            element.style.width = '';
                            element.style.minWidth = '';
                            element.style.maxWidth = '';
                            
                            // Get only the text content (ignoring HTML elements)
                            const textContent = Array.from(element.childNodes)
                                .filter(node => node.nodeType === Node.TEXT_NODE)
                                .map(node => node.textContent.trim())
                                .join(' ')
                                .trim();
                            
                            // Remove width classes first
                            element.classList.remove('too-short-text', 'short-text');
                            
                            // Apply width classes based on content length
                            if (textContent.length <= 40) {
                                element.classList.add('too-short-text');
                            } else if (textContent.length > 40 && textContent.length <= 100) {
                                element.classList.add('short-text');
                            }
                            // Longer text gets no special class (will use max-width)
                        });
                    }
                });
            });
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true
        });


        /**
         * Global Comment Box Management
         */
        function setupCommentBox() {
            const commentBoxContainer = document.querySelector('.comment-box-container');
            if (!commentBoxContainer) return;
            const textarea = document.getElementById('comment_textarea');
            const textareaButtons = document.getElementById('textarea-buttons');
            const actionButtons = document.getElementById('action-buttons');
            // Store original button positions
            const originalButtonHTML = actionButtons.innerHTML;
            
            // Event delegation for action buttons
            document.addEventListener('click', function(e) {
                const button = e.target.closest('.add-comment-btn');
                if (!button) return;
                commentBoxContainer.classList.add('raised');
                textarea.style.display = 'block';
                textareaButtons.style.display = 'flex';
                actionButtons.style.display = 'none';
                textarea.setAttribute('placeholder', button.dataset.placeholder);
                textarea.value = '';
                textarea.focus();
                textarea.dataset.type = button.dataset.type;
            });
            // Cancel button handler
            const cancelBtn = document.getElementById('cancel-comment-btn');
            if (cancelBtn) {
                cancelBtn.addEventListener('click', function() {
                    resetCommentBox();
                });
            }
            // Send button handler
            const sendBtn = document.getElementById('send-comment-btn');
            if (sendBtn) {
                sendBtn.addEventListener('click', function() {
                    if (!textarea.value.trim()) {
                        textarea.focus();
                        textarea.classList.add('error');
                        setTimeout(() => textarea.classList.remove('error'), 1000);
                        return;
                    }
                    resetCommentBox();
                });
            }
                
            // Function to reset the comment box to original state
            function resetCommentBox() {
                commentBoxContainer.classList.remove('raised');
                textarea.style.display = 'none';
                textareaButtons.style.display = 'none';
                actionButtons.style.display = 'flex';
                actionButtons.innerHTML = originalButtonHTML;
                // Reattach event listeners to new buttons
                document.querySelectorAll('.add-comment-btn').forEach(btn => {
                    btn.addEventListener('click', handleCommentButtonClick);
                });
            }
                
            // Separate handler for comment buttons
            function handleCommentButtonClick(e) {

                const textareaHeader = document.getElementById('textarea-header');
                textareaHeader.textContent = this.textContent.trim();
                 // Get the button's color class and apply matching styles
                if (this.classList.contains('red')) {
                    textareaHeader.style.color = '#d9534f'; // Bootstrap danger/red color
                    textareaHeader.style.borderColor = '#d9534f';
                } else if (this.classList.contains('blue')) {
                    textareaHeader.style.color = '#337ab7'; // Bootstrap primary/blue color
                    textareaHeader.style.borderColor = '#337ab7';
                } else if (this.classList.contains('yellow')) {
                    textareaHeader.style.color = '#f0ad4e'; // Bootstrap warning/yellow color
                    textareaHeader.style.borderColor = '#f0ad4e';
                }

                textareaHeader.style.display = 'block';
                textareaHeader.style.backgroundColor = 'white';

                const button = e.currentTarget;
                commentBoxContainer.classList.add('raised');
                textarea.style.display = 'block';
                textareaButtons.style.display = 'flex';
                actionButtons.style.display = 'none';
                textarea.setAttribute('placeholder', button.dataset.placeholder);
                textarea.value = '';
                textarea.focus();
                textarea.dataset.type = button.dataset.type;
            }
            // Initial event listener attachment
            document.querySelectorAll('.add-comment-btn').forEach(btn => {
                btn.addEventListener('click', handleCommentButtonClick);
            });
        }

        window.updateActionButtonsVisibility = function() {
            const orderIdInput = document.getElementById('cmt_order_id');
            const actionButtonsDivs = document.querySelectorAll('#action-buttons');
            const shouldShow = orderIdInput && orderIdInput.value;
            
            actionButtonsDivs.forEach(div => {
                div.style.display = shouldShow ? 'block' : 'none';
            });
        };

        const commentsTab = document.querySelector('.tab_comments');
        if (commentsTab) {
            commentsTab.addEventListener('click', function(e) {
                const orderIdElement = document.getElementById('cmt_order_id');
                if (!orderIdElement || !orderIdElement.value) {
                    e.preventDefault();
                }
            });
        }
        window.updateActionButtonsVisibility();
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initChatbot);
    } else {
        initChatbot();  // DOM is already ready—run now
    }
})();