import Pusher from 'pusher-js';

var isSuperAdmin = true;

(function() {
    function init() {
            // Get current user ID
            const currentUserId = isSuperAdmin ? '1': userId;

            // Initialize Pusher
            const pusher = new Pusher('14ed85e4fc8660f31eb3', {
                cluster: 'us2',
                encrypted: true
            });

            // Subscribe to channel
            const channel = pusher.subscribe('channel-admin');

            // Debug log for channel subscription
            pusher.connection.bind('connected', () => {
                console.log('Connected to Pusher');
            });

            // Bind to new-notification event
            channel.bind('notification', function(payload) {
                if (!payload.data || !payload.data.response) {
                    console.log('Invalid payload structure:', payload);
                    return;
                }
                const notification = payload.data.response;
                if (notification.user_id == currentUserId) {
                    prependNotification({ response: notification, currentUserId: currentUserId });
                } else {
                    console.log('Notification not for current user', {
                        current: currentUserId,
                        target: notification.user_id
                    });
                }
            });

            // Initialize event listeners
            attachHoverEffect();
            attachNotificationTooltipEvents();
            attachPaginationEvents();
            attachCommentEvents();

        function prependNotification(notification, currentUserId) {

            let tbody = document.getElementById('notification-tableBody');
            const container = document.getElementById('notification-container');
            if (!container) return;
            const portalType = container.getAttribute('data-notification-type');
            if (!tbody) {
                    const table = document.createElement('table');
                    table.className = 'table table-striped table-sm table-hover w-auto notification-table';
                    
                    // Create thead
                    const thead = document.createElement('thead');
                    thead.innerHTML = `
                        <tr>
                            <th width="10%"></th>
                            <th width="10%">Order</th>
                            <th width="5%">Tray</th>
                            <th width="5%">Invoice</th>
                            <th width="20%">Patient Reference</th>
                            ${notification.response.portalType === 'backend' ? '<th width="20%">Account</th>' : ''}
                            <th width="10%">Sender</th>
                            <th width="15%">Sent</th>
                            <th width="5%"></th>
                        </tr>
                    `;
                    
                    // Create tbody
                    tbody = document.createElement('tbody');
                    tbody.id = 'notification-tableBody';
                    
                    // Assemble table
                    table.appendChild(thead);
                    table.appendChild(tbody);
                    
                    // Clear "No notifications" message if it exists
                    if (container) {
                        container.innerHTML = '';
                        container.appendChild(table);
                    }
                }

            // Remove "No notifications" message if it exists
            const noNotificationRow = tbody.querySelector('.no-notifications-row');
            if (noNotificationRow) {
                noNotificationRow.remove();
            }


            const row = document.createElement('tr');
            row.className = `notification-item notification-unread`;
            row.setAttribute('data-attr-id', notification.response.id);
            row.setAttribute('data-attr-order-id', notification.response.order_id);
            row.setAttribute('data-attr-log-id', notification.response.plog_id);
            row.setAttribute('data-attr-user-id', notification.response.user_id);
            row.setAttribute('data-attr-type', notification.response.portalType);

            const html = `
                <td width="10%">
                    <div class="notification-icon">
                        <div class="label label-sm label-danger">
                            <i class="fa fa-bell-o" style="font-size: 10px;"></i>
                        </div>
                        <div style="margin-left: 10px;">
                            <span title="${notification.response.portalType}">${notification.response.type_icon}</span>
                        </div>
                    </div>
                </td>
                <td width="10%">${notification.response.order_id_method}</td>
                <td width="5%">${notification.response.tray}</td>
                <td width="5%">${notification.response.invoice_no}</td>
                <td width="20%">${notification.response.patient || ''}</td>
                 ${portalType === 'backend' ? `
                <td width="20%">${notification.response.account}</td>
                ` : ''}
                <td width="10%">${notification.response.sender_name || 'Unknown'}</td>
                <td width="15%">${notification.response.created_at}</td>
                <td width="5%">
                    <div class="notification-tooltip" data-type="read">
                        <i class="fa fa-envelope-o" style="font-size: 10px;"></i>
                        <span class="notification-tooltiptext">Mark as read</span>
                    </div>
                </td>
                `;
            row.innerHTML = html;
            tbody.insertBefore(row, tbody.firstChild);
            console.log('Prepended notification:', notification);
            // Re-attach event listeners
            attachHoverEffect();
            attachNotificationTooltipEvents();
        }


        // Function to handle pagination clicks
        function attachPaginationEvents() {
            document.addEventListener('click', function(e) {
                // Check if the click was on a pagination link
                if (e.target && (e.target.classList.contains('page-link') || e.target.closest('.page-link'))) {
                    e.preventDefault();
                    const paginationLink = e.target.classList.contains('page-link') ? e.target : e.target.closest('.page-link');
                    const url = paginationLink.href;

                    fetch(url)
                        .then(response => response.text())
                        .then(data => {
                            const tempDiv = document.createElement('div');
                            tempDiv.innerHTML = data;
                            const newNotificationContainer = tempDiv.querySelector('#notification-container');
                            const newPagination = tempDiv.querySelector('.pagination');
                            if (newNotificationContainer) {
                                document.getElementById('notification-container').innerHTML = newNotificationContainer.innerHTML;
                            }
                            if (newPagination) {
                                document.querySelector('.pagination').innerHTML = newPagination.innerHTML;
                            }
                            attachHoverEffect();
                            attachNotificationTooltipEvents();
                            document.getElementById('notification-container').scrollTo({
                                top: 0,
                                behavior: 'smooth'
                            });
                        })
                        .catch(error => {
                            console.error('Error fetching paginated data:', error);
                        });
                }
            });
        }

        // Function to handle comment events
        function attachCommentEvents() {
            window.addEventListener('comment-added', function(e) {
                if (e.detail && e.detail.notification) {
                    prependNotification(e.detail.notification);
                }
            });

            window.addEventListener('comment-updated', function(e) {
                if (e.detail && e.detail.notification) {
                    prependNotification(e.detail.notification);
                }
            });

            window.addEventListener('comment-deleted', function(e) {
                if (e.detail && e.detail.notification) {
                    prependNotification(e.detail.notification);
                }
            });
        }

        function attachHoverEffect() {
            const notificationItems = document.querySelectorAll('.notification-item');
            const container = document.getElementById('notification-container');
            if (!container) return;
            const portalType = container.getAttribute('data-notification-type');
            console.log('Portal type:', portalType);
            notificationItems.forEach(item => {
                item.addEventListener('click', () => {
                    let orderId = item.getAttribute('data-attr-order-id');
                    localStorage.setItem('orderId', orderId);
                    const url = `${window.location.origin}/${portalType === 'backend' ? 'admin/orders' : 'orders'}?order=${orderId}&from=notification`;
                    window.location.href = url;
                });
            });
        }

         // Function to attach tooltip events (separate from hover effect)
         function attachNotificationTooltipEvents() {
            const notificationTooltip = document.querySelectorAll('.notification-tooltip');
            
            notificationTooltip.forEach(markIcon => {
                markIcon.addEventListener('click', (event) => {
                    event.stopPropagation();
                    event.stopImmediatePropagation();
                    const tooltip = event.target.closest('.notification-tooltip');
                    if (!tooltip) return;
                    
                    const row = tooltip.closest('tr');
                    if (!row) return;
                    
                    const UID = row.getAttribute('data-attr-id');
                    const isCurrentlyRead = row.classList.contains('notification-read');
                    const newState = isCurrentlyRead ? 'unread' : 'read';
                     
                    fetch('dashboard/mark-notification', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        },
                        body: JSON.stringify({
                            newState: newState,
                            UID: UID
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        markAsRead(row, isCurrentlyRead, tooltip);
                        toastMessage('success', data.message);
                        
                        // Updated counter handling
                        newState === 'read' ? updateNotificationCounter(false) : updateNotificationCounter(true);
                    })
                    .catch(error => {
                        console.log('Error:', error);
                        // toastMessage('error', error.message || 'An error occurred');
                    });
                });
            });
        }

        function updateNotificationCounter(increment = true) {
            // Find or create the badge element
            let countBadge = document.querySelector('#notification-header-link .notifications-badge-counter');
            
            if (!countBadge) {
                // Create the badge if it doesn't exist
                countBadge = document.createElement('span');
                countBadge.className = 'badge badge-danger notifications-badge-counter';
                const notificationLink = document.querySelector('#notification-header-link a');
                if (notificationLink) {
                    notificationLink.appendChild(countBadge);
                }
            }
            
            // Update the count
            let currentCount = parseInt(countBadge.textContent) || 0;
            currentCount = increment ? currentCount + 1 : Math.max(0, currentCount - 1);
            countBadge.textContent = currentCount;
            
            // Hide if count is zero
            if (currentCount === 0) {
                countBadge.style.display = 'none';
            } else {
                countBadge.style.display = 'inline-block';
            }
            return countBadge;
        }

        function markAsRead(row, isCurrentlyRead, tooltip) {
            // Toggle row classes
            row.classList.toggle('notification-read', !isCurrentlyRead);
            row.classList.toggle('notification-unread', isCurrentlyRead);

            // Get the notification icon container
            const notificationIcon = row.querySelector('.notification-icon');
            
            // Toggle grayscale class based on read state
            if (!isCurrentlyRead) {
                notificationIcon.classList.add('grayscale');
            } else {
                notificationIcon.classList.remove('grayscale');
            }

            // 1. Update the envelope icon (mark as read/unread)
            const envelopeIcon = tooltip.querySelector('i.fa-envelope-open-o, i.fa-envelope-o');
            if (envelopeIcon) {
                // Clear both classes first to avoid conflicts
                envelopeIcon.classList.remove('fa-envelope-open-o', 'fa-envelope-o');
                // Apply the correct one
                envelopeIcon.classList.add(isCurrentlyRead ? 'fa-envelope-open-o' : 'fa-envelope-o');
            }

            // 2. Update the status icon (check/bell)
            const statusIcon = row.querySelector('.notification-icon i.fa-check, .notification-icon i.fa-bell-o');
            const statusLabel = row.querySelector('.notification-icon .label');
            if (statusIcon && statusLabel) {
                // Clear both possible icons first
                statusIcon.classList.remove('fa-check', 'fa-bell-o');
                // Add the correct one
                if (!isCurrentlyRead) {
                    statusIcon.classList.add('fa-check');
                    statusLabel.classList.replace('label-danger', 'label-success');
                } else {
                    statusIcon.classList.add('fa-bell-o');
                    statusLabel.classList.replace('label-success', 'label-danger');
                }
            }

            // 3. Update tooltip text
            const tooltipText = tooltip.querySelector('.notification-tooltiptext');
            if (tooltipText) {
                tooltipText.textContent = isCurrentlyRead ? 'Mark as unread' : 'Mark as read';
            }
        }
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();  // DOM is already ready—run now
    }
})();
