(function() {
    function init() {
        let currentPage = 1;
        const recordsPerPage = 200;
        let isLoading = false;
        let nextPage = 2;
        let totalRecords = 0;
        let loadedRecords = 0;
        let pageCache = {};

        // Initialize with server-rendered content
        const initialRows = document.querySelectorAll('#records-container tbody tr');
        loadedRecords = initialRows.length;

        // Set up scroll listener for the scrollable container (parent of #records-container or #records-container itself)
        const recordsContainer = document.querySelector('#records-container');
        const scrollableContainer = recordsContainer ? recordsContainer.parentElement || recordsContainer : null;
        if (scrollableContainer) {
            scrollableContainer.addEventListener('scroll', function () {
                checkScroll();
            });
        }

        // Get total records from hidden field or data attribute
        totalRecords = parseInt(document.querySelector('#total-records')?.value) || 0;

        // Call onPageLoad to handle initial data fetching
        onPageLoad();

        // Check if the container is scrollable
        if (scrollableContainer) {
            scrollableContainer.addEventListener('scroll', checkIfScrollable);
        }

        // Search API code
        $(document).on('click', '#search_button', function (event) {
            event.preventDefault();

            if (isLoading) return;

            const $searchButton = $(this);
            const originalButtonText = $searchButton.html(); // Store original content (icon + text)

            // Disable button and show loading state
            $searchButton.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Loading...');

            const $form = $('#orders_search_form');
            const url = $form.attr('action');
            const data = $form.serializeArray();
            data.push({ name: 'page-no', value: currentPage });
            data.push({ name: 'records-per-page', value: recordsPerPage });

            // Clear cache and reset counters for new search
            pageCache = {};
            nextPage = 1;
            loadedRecords = 0;

            searchOrders(url, $.param(data), $searchButton, originalButtonText);
        });

        // Reset search form
        $(document).on('click', '#reset_search', function(event) {
            event.preventDefault();

            if (isLoading) return;

            const $clearButton = $(this);
            const originalButtonText = $clearButton.html();
            $clearButton.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Clearing...');

            // Clear all text inputs and textareas in the form
            $('#orders_search_form').find("input[type=text], textarea").val("");

            // Clear Flatpickr date range picker
            if (window.dateRangePicker) {
                window.dateRangePicker.clear();
            }

            // Clear hidden date fields
            $('#from_date').val('');
            $('#to_date').val('');

            // Clear and update Select2 dropdowns
            if ($('.customer-static').data('select2')) {
                $('.customer-static').val(null).trigger('change');
            }
            if ($('.order-status-static').data('select2')) {
                $('.order-status-static').val(null).trigger('change');
            }

            // Clear the user type select2 dropdown (specific to user search form)
            if ($('#orders_search_form select[name="status[]"]').data('select2')) {
                $('#orders_search_form select[name="status[]"]').val(null).trigger('change');
            }

            // Clear order info and show default message
            $('#tab_order').html('<center><h3>Choose an order on left to see details...</h3></center>');

            // Make sure the order tab is active
            $('.nav-tabs a[href="#tab_order"]').tab('show');

            // Remove print buttons
            $('.print-btn-container').empty();

            // Scroll order list to top
            if (scrollableContainer) scrollableContainer.scrollTop = 0;

            // --- Directly trigger search ---
            const url = $('#orders_search_form').attr('action');
            const data = $('#orders_search_form').serializeArray();
            data.push({ name: 'page-no', value: currentPage });
            data.push({ name: 'records-per-page', value: recordsPerPage });

            // Clear cache and reset counters for new search
            pageCache = {};
            nextPage = 1;
            loadedRecords = 0;

            // Call searchOrders and pass the clear button to handle loading state
            searchOrders(url, $.param(data), $clearButton, originalButtonText);
        });

        // Check for scrolling to the bottom inside the scrollable container
        function checkScroll() {
            if (!scrollableContainer || isLoading) return;

            const scrollTop = scrollableContainer.scrollTop;
            const containerHeight = scrollableContainer.clientHeight;
            const scrollHeight = scrollableContainer.scrollHeight;

            // Trigger loading at 50% of scroll height
            if (scrollTop + containerHeight >= scrollHeight * 0.5) {
                if (totalRecords === 0 || loadedRecords < totalRecords) {
                    isLoading = true;
                    loadNextPage();
                }
            }
        }

        // Load the next page of records
        function loadNextPage() {
            const scrollPositionBeforeLoad = scrollableContainer.scrollTop;

            const url = $('#orders_search_form').attr('action');
            const data = $('#orders_search_form').serializeArray();
            data.push({ name: 'page-no', value: nextPage });
            data.push({ name: 'records-per-page', value: recordsPerPage });

            $.ajax({
                url: url,
                type: 'POST',
                data: $.param(data),
                success: function (response) {
                    if (response && response.html) {
                        const previousHeight = scrollableContainer.scrollHeight;
                        const tbodyContent = $(response.html).find('tbody').html();

                        if (tbodyContent) {
                            $('#records-container tbody').append(tbodyContent);
                            loadedRecords += response.loadedRecords || recordsPerPage;
                            totalRecords = response.totalRecords || totalRecords;
                            nextPage++;

                            // Update total records hidden field
                            const totalRecordsField = document.querySelector('#total-records');
                            if (totalRecordsField) {
                                totalRecordsField.value = totalRecords;
                            }

                            // Restore scroll position
                            scrollableContainer.scrollTop = scrollPositionBeforeLoad;

                        }
                    }
                    isLoading = false;
                },
                error: function () {
                    isLoading = false;
                },
            });
        }

        // Load records for initial load or search
        function loadRecords() {
            if (isLoading) return;
                isLoading = true;
                const url = $('#orders_search_form').attr('action');
                const data = $('#orders_search_form').serializeArray();
                data.push({ name: 'page-no', value: currentPage });
                data.push({ name: 'records-per-page', value: recordsPerPage });

                $.ajax({
                url: url,
                    type: 'POST',
                    data: $.param(data),
                    success: function (response) {
                        const data = response;
                        const tbodyContent = $(data.html).find('tbody').html();
                        $('#records-container tbody').html(tbodyContent);
                        currentPage = 1;
                        nextPage = 2;
                        totalRecords = data.totalRecords || 0;
                        loadedRecords = data.loadedRecords || 0;

                        // Update total records hidden field
                        const totalRecordsField = document.querySelector('#total-records');
                        if (totalRecordsField) {
                            totalRecordsField.value = totalRecords;
                        }

                        isLoading = false;
                    },
                    error: function () {
                        isLoading = false;
                    },
                });
            }

        // Utility function to get query parameters
        function getQueryParameters() {
            const queryParams = {};
            const searchParams = new URLSearchParams(window.location.search);
            for (const [key, value] of searchParams) {
                queryParams[key] = value;
            }
            return queryParams;
        }

        // Handle page load and initial data fetching
        function onPageLoad() {
            openSelectedOrder();
            if ($('.split-table').length > 0) {
                loadRecords();
            }
        }

        // Check if the container is scrollable
        function checkIfScrollable() {
            if (scrollableContainer) {
                if (scrollableContainer.scrollHeight > scrollableContainer.clientHeight) {
                    scrollableContainer.classList.add('has-scroll');
                } else {
                    scrollableContainer.classList.remove('has-scroll');
                }
            }
        }

        function openSelectedOrder() {
            const orderList = document.querySelector('.order-list');
            if (orderList) {
                const selectedRow = orderList.querySelector('.split-table tr.selected');
                if (selectedRow) {
                    selectedRow.click();
                }
            }
        }

        function searchOrders(url, data, $searchButton, originalButtonText) {
        if (isLoading) return;
        isLoading = true;

        // Fallback for button if not provided
        if (typeof $searchButton === 'undefined') {
            $searchButton = $('#search_button');
            originalButtonText = '<i class="fa fa-search" aria-hidden="true"></i> Search'; // Default text
        }

        $.ajax({
            url: url,
            type: 'POST',
            data: data,
            success: function (response) {
                console.log('DEBUG: order-list.js searchOrders success response:', response);
                if (response && response.html) {
                        const tbodyContent = $(response.html).find('tbody').html();
                        if (tbodyContent) {
                            document.querySelector('#records-container tbody').innerHTML = tbodyContent;
                        } else {
                            console.error('No tbody content found in the response.');
                        }
                } else {
                    console.error('Invalid response format.');
                }

                $searchButton.prop('disabled', false).html(originalButtonText);
                isLoading = false;

                currentPage = 1;
                nextPage = 2;
                totalRecords = response.totalRecords || 0;
                loadedRecords = response.loadedRecords || 0;

                // Update total records hidden field
                const totalRecordsField = document.querySelector('#total-records');
                if (totalRecordsField) {
                    totalRecordsField.value = totalRecords;
                }

                // Auto-scroll to top of scrollable container
                if (scrollableContainer) {
                    scrollableContainer.scrollTop = 0;
                }
            },
            error: function (error) {
                console.error('DEBUG: order-list.js searchOrders error:', error);
                $searchButton.prop('disabled', false).html(originalButtonText);
                isLoading = false;
                console.error('Error during search:', error);
            },
        });
    }
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();  // DOM is already ready—run now
    }
})();